package com.xebialabs.xlrelease.versioning.ascode.scm

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettingsUtil.FOLDER_VERSIONING_ID_ROOT
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettings
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.domain.utils.ScmException
import com.xebialabs.xlrelease.domain.configuration.connector.Repository
import com.xebialabs.xlrelease.scm.connector.JGitConnector
import com.xebialabs.xlrelease.service.{CiIdService, ConfigurationService}
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils

import scala.jdk.CollectionConverters._

/**
 * Separated from FolderVersioningService in order to avoid circular dependency with as-code where it depends on GenerateService
 */
@Service
class FolderVersioningConfigService(ciIdService: CiIdService,
                                    configurationRepository: ConfigurationRepository,
                                    configurationService: ConfigurationService) {

  private val ciType = Type.valueOf(classOf[FolderVersioningSettings])

  def createOrUpdateSettings(newConfig: FolderVersioningSettings): FolderVersioningSettings = {
    var deletedRepo = false
    val config = findSettings(newConfig.getFolderId) match {
      case Some(existingConfig) =>
        if (existingConfig.getGitConnection.getId != newConfig.getGitConnection.getId) {
          deletedRepo = true
          JGitConnector.deleteRepo(existingConfig.getFolderId)
        }
        existingConfig.updateWith(newConfig)
        if (deletedRepo) {
          existingConfig.setAppliedVersion("")
          existingConfig.setAppliedBy("")
          existingConfig.setAppliedDate(null)
          updateSettingsDirectly(existingConfig)
        }
        existingConfig
      case None =>
        newConfig.setId(ciIdService.getUniqueId(ciType, FOLDER_VERSIONING_ID_ROOT))
        newConfig
    }

    configurationService.createOrUpdate(config)
    config
  }

  def deleteSettings(folderId: String): Unit = {
    findSettings(folderId) match {
      case Some(config) => configurationService.delete(config.getId)
      case None =>
    }
  }

  def findSettings(folderId: String): Option[FolderVersioningSettings] =
    configurationRepository.
      findAllByTypeAndTitle[FolderVersioningSettings](ciType, null, folderId, folderOnly = true).asScala.headOption

  def findSettingsByConfigId(configId: String): FolderVersioningSettings = {
    val gitConnection = configurationRepository.find[Repository](configId)
      .getOrElse(throw ScmException(s"Configuration with id $configId not found"))
    val folderVersioningSettings = new FolderVersioningSettings
    folderVersioningSettings.setGitConnection(gitConnection)
    folderVersioningSettings
  }

  def findVersionedFolders: List[FolderVersioningSettings]  = {
   configurationRepository.findAllByType(ciType).asScala.toList.asInstanceOf[List[FolderVersioningSettings]]
  }

  def updateSettingsDirectly(config: FolderVersioningSettings): FolderVersioningSettings = configurationRepository.update(config)

  def validateConfigData(config: FolderVersioningSettings, folderTitle: String): Unit = {
    checkArgument(StringUtils.hasText(config.getScmPath),
      s"Failed to create folder Git versioning configuration for folder $folderTitle. Repository path cannot be empty.")
  }
}
