import React, { useMemo } from 'react';
import 'react-diff-view/style/index.css';
import './react-diff-view.less';

interface ReactDiffViewProps {
    diffString: string;
}

// Helper to determine the type of diff line
const getLineType = (line: string): 'add' | 'remove' | 'context' => {
    if (line.startsWith('+') && !line.startsWith('+++')) return 'add';
    if (line.startsWith('-') && !line.startsWith('---')) return 'remove';
    return 'context';
};

// Helper to remove diff markers for display
const getDisplayLine = (line: string, type: string) => {
    if ((type === 'add' && line.startsWith('+')) || (type === 'remove' && line.startsWith('-'))) {
        return line.slice(1);
    }
    return line;
};

export const ReactDiffView: React.FC<ReactDiffViewProps> = ({ diffString }) => {
    const lines = useMemo(
        () =>
            diffString.split('\n').filter((line, idx, arr) => {
                const trimmed = line.replace(/^[-+]/, '').trim();
                return (
                    !line.startsWith('diff --git') &&
                    !line.startsWith('--- ') &&
                    !line.startsWith('+++ ') &&
                    !line.startsWith('@@') &&
                    trimmed !== '---' && // Exclude lines like '---' or '- ---'
                    !(idx === arr.length - 1 && line.trim() === '')
                );
            }),
        [diffString],
    );

    let oldLineNum = 1;
    let newLineNum = 1;

    return (
        <pre className="custom-diff-view language-yaml">
            {lines.map((line, idx) => {
                const type = getLineType(line);

                let oldNum = '';
                let newNum = '';

                if (type === 'add') {
                    newNum = `${newLineNum++}`;
                } else if (type === 'remove') {
                    oldNum = `${oldLineNum++}`;
                } else {
                    oldNum = `${oldLineNum++}`;
                    newNum = `${newLineNum++}`;
                }

                const displayLine = getDisplayLine(line, type);

                return (
                    <div className={`diff-line diff-line-${type}`} key={idx}>
                        <span className="diff-line-number">{oldNum}</span>
                        <span className="diff-line-number">{newNum}</span>
                        <span>{displayLine}</span>
                    </div>
                );
            })}
        </pre>
    );
};
