package com.xebialabs.xlrelease.ascode.service.referencehandler.stringhandler

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.utils.TypeSugar.typeOf
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.deployit.repository.ItemConflictException
import com.xebialabs.xlrelease.ascode.service.FolderAsCodeService
import com.xebialabs.xlrelease.ascode.service.referencehandler.stringhandler.TemplateFilterStringReferenceCreationHandler.templateIdDescriptor
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.service.ReleaseService
import com.xebialabs.xlrelease.udm.reporting.filters.impl.TemplateFilter
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

object TemplateFilterStringReferenceCreationHandler {
  val templateField = "templateId"

  def templateIdDescriptor: PropertyDescriptor = typeOf[TemplateFilter].getDescriptor.getPropertyDescriptor(templateField)

  val properties: Set[String] = Set(
    templateField
  )
}

@Component
class TemplateFilterStringReferenceCreationHandler @Autowired()(releaseService: ReleaseService,
                                                                folderAsCodeService: FolderAsCodeService) extends StringReferenceCreationHandler {

  override def isDefinedAt(config: StringReferenceCreationConfig): Boolean = {
    config.ci.isInstanceOf[TemplateFilter] &&
      (templateIdDescriptor.get(config.ci) != null)
  }

  override def apply(config: StringReferenceCreationConfig): StringReferenceContainer = {
    val templateId = Option(templateIdDescriptor.get(config.ci)) match {
      case Some(id: String) =>
        val pathAndTitle = Utils.parseAbsolutePath(id)

        val folderId = pathAndTitle.folderPath
          .filter(currentId => currentId.nonEmpty)
          .flatMap(path => folderAsCodeService.searchFolder(FolderAsCodeService.absolutePath(path, config.home)).map(_.getId))
          .orNull

        val templateName = pathAndTitle.title
        val templates = releaseService.findTemplatesByTitle(folderId, templateName, 0, Integer.MAX_VALUE, Integer.MAX_VALUE).asScala
        if (templates.nonEmpty && templates.size > 1) {
          throw new ItemConflictException(s"More than one template with title [$templateName] is present in the folder [$folderId]. Please use an unique name.")
        }
        val tId = templates.headOption.map(t => Ids.getFolderlessId(t.getId))
        if (tId.isEmpty) {
          throw new AsCodeException(s"Template with path [$id] was not found")
        }
        tId
      case _ => Option.empty
    }

    StringReferenceContainer(config.ci, Map(templateIdDescriptor -> templateId))
  }
}
