package com.xebialabs.xlrelease.versioning.ascode

import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.versioning.ascode.CiInfo
import com.xebialabs.xlrelease.domain.versioning.ascode.validation.{DuplicateTitle, FolderInfo, ValidationMessage}

import scala.jdk.CollectionConverters._

object DuplicateTitlesValidator {

  def validate(definitions: Seq[Definition], rootFolder: FolderInfo): Seq[ValidationMessage] = {
    definitions.flatMap {
      case Definition(_, _, _, spec: CiSpec) => validateCis(spec.cis, rootFolder)
      case _ => Seq.empty
    }
  }

  private def validateCis(cis: List[ConfigurationItem], parentFolder: FolderInfo): Seq[ValidationMessage] = {
    val duplicateMessages = findDuplicates(cis, parentFolder)
    val nestedMessages = cis.collect { case f: Folder =>
      val nested = f.getChildren.asInstanceOf[java.util.Set[ConfigurationItem]].asScala.toList
      val relPath = if (parentFolder.getRelativePath.isEmpty) f.getTitle else s"${parentFolder.getRelativePath}/${f.getTitle}"
      val absPath = s"${parentFolder.getAbsolutePath}/${f.getTitle}"
      val childFolder = new FolderInfo(f.getId, relPath, absPath)
      validateCis(nested, childFolder)
    }.flatten

    duplicateMessages ++ nestedMessages
  }

  private def getTitle(ci: ConfigurationItem): Option[String] = {
    if (!ci.hasProperty("title")) {
      Option.empty[String]
    }
    else {
      val title: String = ci.getProperty("title")
      if (title.isEmpty) Option.empty[String] else Some(title)
    }
  }

  private def findDuplicates(cis: List[ConfigurationItem], parentFolder: FolderInfo): Seq[ValidationMessage] = {
    val titleCiPairs = cis.flatMap { ci =>
      getTitle(ci).map(title => (title, ci))
    }

    val duplicateGroups = titleCiPairs
      .groupBy { case (title, _) => title }
      .filter { case (_, pairs) => pairs.size > 1 }

    val duplicateValidationMessages = duplicateGroups.values.flatten
      .map { case (_, ci) =>
        new DuplicateTitle(CiInfo(ci, parentFolder))
      }

    duplicateValidationMessages.toSeq
  }
}
