package com.xebialabs.xlrelease.ascode.`export`

import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind
import com.xebialabs.deployit.plumbing.`export`.UnresolvedReferencesConfigurationItemConverter._
import com.xebialabs.xlrelease.ascode.service.{ReferenceSolver, TemplateAsCodeService}
import com.xebialabs.xlrelease.ascode.utils.{GlobalScope, ImportContext, TemplateScope}
import com.xebialabs.xlrelease.ascode.yaml.parser.XLRDefinitionParser
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.utils.CiHelper.getNestedCis

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}


trait YamlTemplateImportHelper {
  def parser: XLRDefinitionParser

  def ciIdService: CiIdService

  def referenceSolver: ReferenceSolver

  def templateAsCodeService: TemplateAsCodeService

  private val wrongCiMessage = "Only templates are allowed for import."
  private val TAGGED_PASSWORD = "TAGGED_PASSWORD"


  def parseAndValidate(input: String): Definition = {
    val inputWithoutTaggedPasswords = replaceTaggedPassword(input)
    val parseResult = Try(parser.parse(inputWithoutTaggedPasswords))

    parseResult match {
      case Failure(exception) =>
        throw new IllegalArgumentException(exception.getMessage)
      case Success(definition) =>
        validateYamlDefinition(definition)
        definition
    }
  }

  def convertDefinitionToRelease(definition: Definition, destinationFolderId: String, newReleaseId: Option[String] = None): Release = {
    definition.spec match {
      case spec: CiSpec =>
        spec.cis.head match {
          case release: Release =>
            referenceSolver.resolveReferences(release, spec.references, destinationFolderId)
            resetPasswordsIfTagged(release)
            val id = newReleaseId.getOrElse(ciIdService.getUniqueId(release.getType, destinationFolderId))
            templateAsCodeService.populateTemplateData(
              ImportContext(scope = TemplateScope(id, release.getTitle, GlobalScope)),
              release,
              id
            )
            release
          case _ =>
            throw new IllegalArgumentException(wrongCiMessage)
        }
      case _ =>
        throw new IllegalArgumentException(wrongCiMessage)
    }
  }

  private def validateYamlDefinition(definition: Definition): Unit = {
    if (!definition.kind.equals("Templates")) {
      throw new IllegalArgumentException(wrongCiMessage)
    }
    definition.spec match {
      case spec: CiSpec =>
        if (spec.cis.length > 1) {
          throw new IllegalArgumentException("The YAML file includes more than one configuration item. Only one template is allowed per import.")
        } else if (!spec.cis.head.isInstanceOf[Release]) {
          throw new IllegalArgumentException(wrongCiMessage)
        }
      case _ =>
        throw new IllegalArgumentException(wrongCiMessage)
    }
  }

  private def resetPasswordsIfTagged(template: Release): Unit = {
    val configurationItems = getNestedCis(template)
    configurationItems.forEach { configurationItem =>
      configurationItem.getType.getDescriptor.getPropertyDescriptors.asScala.foreach { propertyDescriptor =>
        if (propertyDescriptor.getKind == PropertyKind.STRING && propertyDescriptor.isPassword) {
          val currentPassword = propertyDescriptor.get(configurationItem).asInstanceOf[String]
          if (currentPassword == TAGGED_PASSWORD) {
            propertyDescriptor.set(configurationItem, DECRYPTION_FAILED_VALUE)
          }
        }
      }
    }
  }

  private def replaceTaggedPassword(input: String): String = {
    input.replaceAll(": !value \".+\"", s""": $TAGGED_PASSWORD""")
  }

}
