package com.xebialabs.xlrelease.analytics.service

import com.xebialabs.deployit.security.PermissionEnforcer
import com.xebialabs.xlrelease.api.v1.forms.{ReleaseOrderDirection, ReleaseOrderMode, ReleasesFilters, TimeFrame}
import com.xebialabs.xlrelease.api.v1.{ConfigurationApi, FolderApi}
import com.xebialabs.xlrelease.db.sql.SqlBuilder
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.archiving.SelectArchivedReleasesBuilder
import com.xebialabs.xlrelease.domain.analytics.ProjectedRelease
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.reports.filters.CompositeFilter.Operator.OR
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.reports.service.ReportsService
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.ReleasePersistence
import com.xebialabs.xlrelease.repository.{Ids, Page, ReleaseRepository}
import com.xebialabs.xlrelease.service.SqlReleasesFilterSupport
import com.xebialabs.xlrelease.udm.reporting.AuditReportRequest
import com.xebialabs.xlrelease.udm.reporting.filters.SelectArchivedReleasesBuilderVisitor
import com.xebialabs.xlrelease.udm.reporting.filters.impl.{DateFilter, ReleaseStatusFilter}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Service

import java.util.Date
import scala.jdk.CollectionConverters._

@Service
class PlannerService @Autowired()(releaseRepository: ReleaseRepository,
                                  releasePersistence: ReleasePersistence,
                                  reportsService: ReportsService,
                                  @Qualifier("xlrRepositorySqlDialect") implicit val sqlDialect: Dialect,
                                  implicit val permissionEnforcer: PermissionEnforcer,
                                  configurationApi: ConfigurationApi,
                                  folderApi: FolderApi) extends Logging {

  def planActiveReleases(page: Long, numberByPage: Long): Seq[ProjectedRelease] = {
    val filters = new ReleasesFilters
    filters.setPlanned(true)
    filters.setActive(true)
    filters.setOrderBy(ReleaseOrderMode.start_date)
    filters.setOrderDirection(ReleaseOrderDirection.ASC)

    val sqlWithParams = SqlReleasesFilterSupport.sqlBuilderByFilters(filters, Seq.empty, Seq.empty)
      .selectReleaseId()
      .withPage(Page(page.toInt, numberByPage.toInt, 0))
      .build()

    val releaseIds = releasePersistence.findReleaseIdsByQuery(sqlWithParams)
    val plannedReleases = planReleases(releaseIds.toSeq)
    plannedReleases
  }

  def planCompletedReleases(page: Long, numberByPage: Long, lastChecked: Date): Seq[ProjectedRelease] = {
    val request = AuditReportRequest(
      Seq(
        new CompFilter(OR,
          new ReleaseStatusFilter(ReleaseStatus.COMPLETED),
          new ReleaseStatusFilter(ReleaseStatus.ABORTED)
        ),
        new DateFilter(TimeFrame.RANGE, lastChecked, null)
      ), Some(ReleaseOrderMode.start_date))

    val releases = reportsService.getReleasesForPreview(request, page, numberByPage).asScala.toList
    val calculator = new PlanCalculator(null) // No need for variable resolution in completed releases
    val computed = calculator.calculateDates(releases)
    releases.map(rel => computed(rel.getId).asInstanceOf[ProjectedRelease])
  }

  private def releaseIdsWithDependencies(releaseIds: List[String]): List[String] = {
    var processing: Set[String] = releaseIds.map(Ids.getName).toSet
    var releases: Set[String] = Set.empty

    while (!processing.isEmpty) {
      val targets = releasePersistence.getDependencies(processing.toList).map(_.targetReleaseId).toSet
      releases ++= processing
      processing = if (targets.isEmpty) Set.empty else targets diff releases
    }
    releases.toList
  }

  def planReleases(releaseIds: Seq[CiId]): Seq[ProjectedRelease] = {
    val relsWithDeps = releaseIdsWithDependencies(releaseIds.toList)
    val calculator = new PlanCalculator(new VariableResolver(configurationApi, folderApi))
    val computed = calculator.calculateDates(releaseRepository.getReleasesWithoutComments(relsWithDeps))
    releaseIds.map(id => computed(id).asInstanceOf[ProjectedRelease])
  }
}

class CompFilter(operator: com.xebialabs.xlrelease.reports.filters.CompositeFilter.Operator, filters: ReportFilter*) extends
  com.xebialabs.xlrelease.udm.reporting.filters.CompositeFilter(operator, filters: _*) {

  def this() = {
    this(null)
  }

  override def apply[T <: SqlBuilder[T]](builder: T): Unit = {
    builder match {
      case b: SelectArchivedReleasesBuilder => new CustomSelectArchivedReleasesBuilderVisitor(b)
      case b => super.apply(b)
    }
  }
}

class CustomSelectArchivedReleasesBuilderVisitor(override val builder: SelectArchivedReleasesBuilder)
  extends SelectArchivedReleasesBuilderVisitor(builder) {
  override def visit(filter: ReleaseStatusFilter): Unit = {
    if (filter.getReleaseStatus != null) {
      builder.withOneOfStatuses(filter.getReleaseStatus.value())
    }
  }
}

