/* istanbul ignore file */

import { useEffect, useState } from 'react';

export const useMicrostrategy = () => {
    const [loginPageLoaded, setLoginPageLoaded] = useState<boolean>(false);
    const [mstrToken, setMstrToken] = useState<string | null>(null);
    const [newTab, setNewTab] = useState<WindowProxy | null>(null);
    const [mstrBaseUrl, setMstrBaseUrl] = useState<string>();
    const [mstrError, setMstrError] = useState<Error | null>(null);

    //Fetch auth token
    function getAuthToken(mstrUrl: string) {
        const options: RequestInit = {
            method: 'GET',
            credentials: 'include', // Including cookie
            mode: 'cors', // Setting as cors mode for cross origin
            headers: { 'content-type': 'application/json' },
        };
        fetch(mstrUrl + '/Library/api/auth/token', options)
            .then((response) => {
                if (response.ok) {
                    const token = response.headers.get('x-mstr-authtoken');
                    setMstrToken(token);
                    setMstrError(null);
                } else {
                    // Log the response json if it failed
                    setMstrError(Error('Auth failed for a backed analytics service. Complete functionality may not be available.\n'));
                    setMstrToken(null);
                }
            })
            .catch((error) => {
                // Log the error if failed
                setMstrError(error);
                setMstrToken(null);
            });
    }

    // Event Listener method the marks the login page as loaded, we can't close it til this is complete.
    function loginPageCallback() {
        //OIDC auth is complete
        window.removeEventListener('message', loginPageCallback);
        setLoginPageLoaded(true);
    }

    // After open the MSTR login page, if it's loaded close the tab and reset the login .
    useEffect(() => {
        if (loginPageLoaded && mstrBaseUrl) {
            newTab?.close();
            setNewTab(null);
            getAuthToken(mstrBaseUrl);
        }
    }, [loginPageLoaded, mstrBaseUrl]);

    return {
        mstrError,
        mstrToken,
        // Called to kick off an OIDC login into MSTR.  Either the mstrToken or mstrError will be set
        // once login is complete.
        mstrLogin: (mstrUrl: string) => {
            if (mstrUrl !== mstrBaseUrl) {
                setMstrBaseUrl(mstrUrl);
            }

            // Set a listener so know when page has been loaded
            window.addEventListener('message', loginPageCallback);

            // loginMode signaling an OIDC login
            const oidcLoginMode = 4194304;

            const loginUrl =
                mstrUrl + '/Library/auth/login-dialog.jsp?loginMode=' + oidcLoginMode + '&callback-origin=' + encodeURIComponent(window.location.origin);

            //need to store the tab. close it later.
            const tab = window.open(loginUrl, '_blank');
            setNewTab(tab);
        },
    };
};
