import React from 'react';
import { DotAvatar, DotBadge, DotButton, DotChip, DotIcon, DotIconButton, DotInputText, DotTabs, DotTypography } from '@digital-ai/dot-components';
import { mockResizeObserver, mountWithStoreAndTheme, ReactWrapper } from '../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { AnalyticsPage, AnalyticsPageProps } from './analytics-page.component';
import { AnalyticsCards } from './analytics-cards.component';
import { ANALYTICS_CATEGORIES, ANALYTICS_SAMPLE_CARDS, DIGITAL_AI_AUTHOR } from '../constants';
import { AnalyticsPageFilterDrawer } from './analytics-page-filter-drawer.component';
import { RootState } from '../../../../../../../../../core/xlr-ui/app/js/store.types';
import { analytics, AnalyticsPageFilter, AnalyticsTabsEnum } from '../ducks/analytics.reducer';
import { AnalyticsPagePortal } from './analytics-page.portal';
import { mockIntersectionObserver } from '../../../../../../../../../core/xlr-ui/app/features/release-stepper/__mocks__';
import { DEFAULT_PAGINATION } from '../../../../../../../../../core/xlr-ui/app/constants/pagination';

const { filterCards, setIsDrawerOpened, init, favoriteDashboardFromList, setSelectedTab, resetAnalytics } = analytics.actions;

describe('AnalyticsPage', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();
    const authors = [DIGITAL_AI_AUTHOR, 'Custom'];
    const filteredAuthorNames = [DIGITAL_AI_AUTHOR];
    const filteredCategories = [ANALYTICS_CATEGORIES[0]];
    const cardSearch = { categories: filteredCategories, authorNames: filteredAuthorNames };
    const categories = ANALYTICS_CATEGORIES;
    const numberOfFiltersApplied = 2;
    const defaultPagination = DEFAULT_PAGINATION;
    const allCards = ANALYTICS_SAMPLE_CARDS;
    const favoriteCards = [ANALYTICS_SAMPLE_CARDS[5], ANALYTICS_SAMPLE_CARDS[6]];

    const defaultState: RootState = {
        analytics: {
            authors,
            allCards,
            favoriteCards,
            cardSearch,
            categories,
            isDrawerOpened: false,
            isLoading: false,
            intelligenceConfiguration: undefined,
            manageDashboardUrl: 'url',
            totalAvailableCards: allCards.length,
            totalAvailableFavoriteCards: favoriteCards.length,
            updatingDashboardIds: [],
        },
        licenseWarning: { license: { edition: 'Premium' } },
    };

    const filters: AnalyticsPageFilter = {
        name: 'initial filter',
    };

    const defaultProps: AnalyticsPageProps = {
        filters,
    };

    const mount = (state = defaultState, props = defaultProps) => {
        wrapper = mountWithStoreAndTheme(<AnalyticsPage {...props} />, dispatch, state);
    };

    const getAlertBanner = () => wrapper.find('div.analytics-banner');
    const getAnalyticsBannerAvatar = () => getAlertBanner().find(DotAvatar);
    const getAnalyticsBannerTypography = () => getAlertBanner().find(DotTypography);
    const getPageWrapper = () => wrapper.find('div.analytics-page');
    const getPageTitle = () => wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'page-title');
    const getDotTabs = () => wrapper.find(DotTabs);
    const getAnalyticsCards = () => wrapper.find(AnalyticsCards);

    const getClearAllButton = () => wrapper.findWhere((node) => node.is(DotButton) && node.props()['data-testid'] === 'clear-all-btn');
    const getContentHeaderFilterDiv = () => wrapper.find('div.content-header-filter');
    const getNameFilterInput = () => getContentHeaderFilterDiv().find(DotInputText);
    const getFilterBadge = () => getContentHeaderFilterDiv().find(DotBadge);
    const getFilterBadgeIconButton = () => getFilterBadge().find(DotIconButton);
    const getNameFilterInputIcon = () =>
        getNameFilterInput().findWhere((node) => node.is(DotIcon) && node.props()['data-testid'] === 'name-filter-search-icon');

    const getHeaderFiltersRow = () => wrapper.findWhere((node) => node.is('div.aligned-flex-with-gap') && node.props()['data-testid'] === 'filters-row');
    const getCategoriesFilterTitleTypography = () =>
        wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'category-filters-title');
    const getAuthorFilterTitleTypography = () => wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'author-filters-title');
    const getAnalyticsPageFilterDrawer = () => wrapper.find(AnalyticsPageFilterDrawer);
    const getFilterChips = (startsWithId: string) =>
        getHeaderFiltersRow().findWhere((node) => node.is(DotChip) && node.props()['data-testid'].startsWith(startsWithId));
    const getCategoriesFilterChips = () => getFilterChips('category-filters');
    const getAuthorFilterChips = () => getFilterChips('author-filters');
    const filterByName = (text: string) => getNameFilterInput().invoke('onChange')?.({ target: { value: text } } as never);
    const getPortal = () => wrapper.find(AnalyticsPagePortal);

    beforeEach(() => {
        mockResizeObserver();
        mockIntersectionObserver(jest.fn(), jest.fn());
    });

    afterEach(() => jest.resetAllMocks());

    it('should render page with proper elements', () => {
        mount();
        expect(getPageWrapper()).toExist();

        const pageTitle = getPageTitle();
        expect(pageTitle).toExist();
        expect(pageTitle.props().variant).toBe('h1');
        expect(pageTitle.props().children).toBe('Analytics');

        const nameFilterInput = getNameFilterInput();
        expect(nameFilterInput).toExist();
        const nameFilterInputProps = nameFilterInput.props();
        expect(nameFilterInputProps.className).toBe('name-filter');
        expect(nameFilterInputProps.defaultValue).toBe('initial filter');
        expect(nameFilterInputProps.id).toBe('name-filter');
        expect(nameFilterInputProps.placeholder).toBe('Filter by name ...');
        expect(nameFilterInputProps.onClear).toStrictEqual(expect.any(Function));

        const icon = getNameFilterInputIcon();
        expect(icon).toExist();
        expect(icon.props().fontSize).toBe('small');
        expect(icon.props().iconId).toBe('search');

        const filterBadge = getFilterBadge();
        expect(filterBadge).toExist();
        const filterBadgeProps = filterBadge.props();
        expect(filterBadgeProps.badgeColor).toBe('#d61f21');
        expect(filterBadgeProps.badgeContent).toBe(numberOfFiltersApplied);
        expect(filterBadgeProps.overlap).toBe('circular');
        expect(filterBadgeProps.variant).toBe('standard');

        const filterBadgeIconButton = getFilterBadgeIconButton();
        expect(filterBadgeIconButton).toExist();
        expect(filterBadgeIconButton.props().iconId).toBe('filter');

        const categoriesFilterTitleTypography = getCategoriesFilterTitleTypography();
        expect(categoriesFilterTitleTypography).toExist();
        expect(categoriesFilterTitleTypography.props().variant).toBe('subtitle2');
        expect(categoriesFilterTitleTypography.props().children).toBe('Categories:');

        const categoriesFilterChipGroup = getCategoriesFilterChips();
        expect(categoriesFilterChipGroup).toExist();
        expect(categoriesFilterChipGroup).toIncludeText(filteredCategories.join(''));

        const authorFilterTitleTypography = getAuthorFilterTitleTypography();
        expect(authorFilterTitleTypography).toExist();
        expect(authorFilterTitleTypography.props().variant).toBe('subtitle2');
        expect(authorFilterTitleTypography.props().children).toBe('Authors:');

        const authorsFilterChipGroup = getAuthorFilterChips();
        expect(authorsFilterChipGroup).toExist();
        expect(authorsFilterChipGroup).toIncludeText(filteredAuthorNames.join(''));
        const firstAuthorChip = authorsFilterChipGroup.at(0);
        expect(firstAuthorChip.props().avatar).toBeDefined();

        const tabs = getDotTabs();
        expect(tabs).toExist();
        expect(tabs.props().className).toBe('analytics-tabs');
        expect(tabs.props().tabs).toStrictEqual([{ label: `All dashboards (${allCards.length})` }, { label: `Favorites (${favoriteCards.length})` }]);

        const portal = getPortal();
        expect(portal).toExist();
        expect(portal.props().buttonHref).toBe(defaultState.analytics.manageDashboardUrl);

        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards).toExist();
        expect(analyticsCards.props().cards).toStrictEqual(allCards);
        expect(analyticsCards.props().isFavoritePage).toStrictEqual(false);
        expect(analyticsCards.props().isAnalyticsConfigured).toStrictEqual(false);

        const analyticsPageFilterDrawer = getAnalyticsPageFilterDrawer();
        expect(analyticsPageFilterDrawer).toExist();
        const analyticsPageFilterDrawerProps = analyticsPageFilterDrawer.props();
        expect(analyticsPageFilterDrawerProps.authors).toStrictEqual(authors);
        expect(analyticsPageFilterDrawerProps.cardSearch).toStrictEqual(cardSearch);
        expect(analyticsPageFilterDrawerProps.categories).toStrictEqual(categories);
        expect(analyticsPageFilterDrawerProps.isDrawerOpened).toStrictEqual(false);
        expect(analyticsPageFilterDrawerProps.numberOfFiltersApplied).toStrictEqual(numberOfFiltersApplied);

        expect(dispatch).toHaveBeenCalledWith(init(filters));
    });

    it('should pass folderId prop to card if present', () => {
        mount(defaultState, { ...defaultProps, folder: { id: 'Applications/Folder1' } as never });

        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards.props().folderId).toBe('Applications/Folder1');
    });

    it('should not render filter header row when no filters present', () => {
        const customState = { ...defaultState, analytics: { ...defaultState.analytics, cardSearch: undefined } };
        mount(customState);
        expect(getHeaderFiltersRow()).not.toExist();
    });

    it('should render clear all button and handle event', () => {
        mount();
        expect(getHeaderFiltersRow()).toExist();
        expect(getClearAllButton()).toExist();
        getClearAllButton().invoke('onClick')?.();
        expect(dispatch).toHaveBeenCalledWith(
            filterCards({
                name: undefined,
                categories: undefined,
                authorNames: undefined,
                ...defaultPagination,
            }),
        );
    });

    it('should render filter header row with only Categories filters', () => {
        const customState = {
            ...defaultState,
            analytics: { ...defaultState.analytics, cardSearch: { categories: [categories[0]] } },
        };
        mount(customState);
        expect(getHeaderFiltersRow()).toExist();
        expect(getCategoriesFilterTitleTypography()).toExist();
        expect(getCategoriesFilterChips().length).toBe(1);
        expect(getAuthorFilterTitleTypography()).not.toExist();
        expect(getAuthorFilterChips()).not.toExist();
    });

    it('should render filter header row with only Author filters', () => {
        const customState = {
            ...defaultState,
            analytics: { ...defaultState.analytics, cardSearch: { authorNames: [authors[0]] } },
        };
        mount(customState);
        expect(getHeaderFiltersRow()).toExist();
        expect(getAuthorFilterTitleTypography()).toExist();
        expect(getAuthorFilterChips().length).toBe(1);
        expect(getCategoriesFilterTitleTypography()).not.toExist();
        expect(getCategoriesFilterChips()).not.toExist();
    });

    it('should execute correct handlers when events are invoked', () => {
        mount();
        const currentCardSearch = { categories: [categories[0]], ...defaultPagination };
        const analyticsPageFilterDrawer = getAnalyticsPageFilterDrawer();
        analyticsPageFilterDrawer.invoke('onCardFilter')(currentCardSearch);
        expect(dispatch).toHaveBeenLastCalledWith(filterCards(currentCardSearch));

        analyticsPageFilterDrawer.invoke('onClearAll')();
        expect(dispatch).toHaveBeenLastCalledWith(filterCards({ ...defaultPagination }));

        analyticsPageFilterDrawer.invoke('onDrawerClose')();
        expect(dispatch).toHaveBeenLastCalledWith(setIsDrawerOpened(false));
    });

    it('should open the drawer if is not open', () => {
        const stopPropagation = jest.fn();
        mount({ ...defaultState, analytics: { ...defaultState.analytics, isDrawerOpened: false } });
        getFilterBadgeIconButton().invoke('onClick')?.({ stopPropagation } as never);
        expect(dispatch).toHaveBeenCalledWith(setIsDrawerOpened(true));
        expect(stopPropagation).toHaveBeenCalled();
    });

    it('should not open the drawer if is open', () => {
        const stopPropagation = jest.fn();
        mount({ ...defaultState, analytics: { ...defaultState.analytics, isDrawerOpened: true } });
        getFilterBadgeIconButton().invoke('onClick')?.({ stopPropagation } as never);
        expect(dispatch).not.toHaveBeenCalledWith(setIsDrawerOpened(true));
        expect(stopPropagation).not.toHaveBeenCalled();
    });

    it('should handle categories chip delete event', () => {
        mount();
        const chip = getCategoriesFilterChips().at(0);
        chip.invoke('onDelete')?.();
        expect(dispatch).toHaveBeenLastCalledWith(filterCards({ authorNames: filteredAuthorNames, categories: [], isFavorite: false, ...defaultPagination }));
    });

    it('should handle author chip delete event', () => {
        mount();
        const chip = getAuthorFilterChips().at(0);
        chip.invoke('onDelete')?.();
        expect(dispatch).toHaveBeenLastCalledWith(filterCards({ authorNames: [], categories: filteredCategories, isFavorite: false, ...defaultPagination }));
    });

    it('should execute correct handler when filtering by name', () => {
        const searchText = 'abc123';
        mount();
        filterByName(searchText);
        expect(dispatch).toHaveBeenLastCalledWith(filterCards({ ...cardSearch, name: searchText, isFavorite: false, ...defaultPagination }));
    });

    it('should execute correct handler when clearing name filter', () => {
        mount();
        filterByName('abc123');
        const nameInputFilter = getNameFilterInput();
        nameInputFilter.invoke('onClear')();
        expect(dispatch).toHaveBeenLastCalledWith(filterCards({ ...cardSearch, name: '', isFavorite: false, ...defaultPagination }));
    });

    it('should trigger correct action when onChange has been invoked on DotTabs', () => {
        mount();
        const selectedTab = AnalyticsTabsEnum.Favorites;
        getDotTabs().invoke('onChange')(selectedTab);
        expect(dispatch).toHaveBeenLastCalledWith(setSelectedTab(selectedTab));
    });

    it('should render Favorites page', () => {
        mount({ ...defaultState, analytics: { ...defaultState.analytics, selectedTab: AnalyticsTabsEnum.Favorites } });

        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards).toExist();
        expect(analyticsCards.props().cards).toStrictEqual(favoriteCards);
        expect(analyticsCards.props().isFavoritePage).toStrictEqual(true);
    });

    it('should NOT render AnalyticsCards when main isLoading prop is set to true', () => {
        mount({ ...defaultState, analytics: { ...defaultState.analytics, isLoading: true } });
        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards).not.toExist();
        expect(getDotTabs()).toExist();
    });

    it('should render tabs without numbers when isLoading', () => {
        mount({ ...defaultState, analytics: { ...defaultState.analytics, isLoading: true } });
        expect(getDotTabs().props().tabs).toStrictEqual([{ label: 'All dashboards' }, { label: 'Favorites' }]);
    });

    it('should trigger analytics reset on unmount', () => {
        mount();
        wrapper.unmount();
        expect(dispatch).toHaveBeenLastCalledWith(resetAnalytics());
    });

    it('should execute correct action when onFavoriteDashboard has been invoked', () => {
        const id = '123';
        mount();
        getAnalyticsCards().invoke('onFavoriteDashboard')?.(id, false);
        expect(dispatch).toHaveBeenLastCalledWith(favoriteDashboardFromList({ id, isUnfavorite: false }));
    });

    describe('license', () => {
        const assertBannerExistence = () => {
            const dotAlertBanner = getAlertBanner();
            expect(dotAlertBanner).toExist();

            const dotAvatar = getAnalyticsBannerAvatar();
            expect(dotAvatar).toExist();
            const dotAvatarProps = dotAvatar.props();
            expect(dotAvatarProps.alt).toBe('crown icon');
            expect(dotAvatarProps.className).toBe('analytics-banner-avatar');
            expect(dotAvatarProps.iconId).toBe('crown');

            const analyticsBannerTypography = getAnalyticsBannerTypography();
            expect(analyticsBannerTypography).toExist();
            const analyticsBannerTypographyProps = analyticsBannerTypography.props();
            expect(analyticsBannerTypographyProps.variant).toBe('body1');
            expect(analyticsBannerTypographyProps.children).toBe(
                'Analytics dashboards are available in the Premium edition. For more information, please contact your Customer Success Manager.',
            );
        };

        it('should render banner when license is Premium edition', () => {
            mount({ ...defaultState, licenseWarning: { license: { edition: 'Premium' } } });
            assertBannerExistence();
        });

        it('should render banner when license is Essentials edition', () => {
            mount({ ...defaultState, licenseWarning: { license: { edition: 'Essentials' } } });
            assertBannerExistence();
        });

        it('should render banner when license is Pro edition', () => {
            mount({ ...defaultState, licenseWarning: { license: { edition: 'Pro' } } });
            assertBannerExistence();
        });

        it('should render banner when license is Trial edition', () => {
            mount({ ...defaultState, licenseWarning: { license: { edition: 'Pro' } } });
            assertBannerExistence();
        });

        it('should NOT render banner when license is NOT Premium edition', () => {
            mount({ ...defaultState, licenseWarning: { license: { edition: 'Community' } } });
            const dotAlertBanner = getAlertBanner();
            expect(dotAlertBanner).not.toExist();
        });

        it('should NOT render banner when license is Premium edition BUT the server is configured', () => {
            mount({
                ...defaultState,
                analytics: {
                    ...defaultState.analytics,
                    intelligenceConfiguration: {
                        folderColumnName: 'test',
                    },
                },
                licenseWarning: { license: { edition: 'Premium' } },
            });
            const dotAlertBanner = getAlertBanner();
            expect(dotAlertBanner).not.toExist();
        });
    });
});
