import {
    generateEnvironmentTitle,
    getCommitHashFromCommitUrl,
    getDeepLink,
    getDiscoveryCardSubheader,
    getDiscoveryCardTitle,
    getFilterCardTitle,
    getServerIcon,
    getTrimmedCommitHashFromCommitUrl,
    getVersionLabel,
    getVersionLink,
    getVersionTooltip,
    mapOrderByValue,
} from './utils';
import {
    APPLICATION_COLUMN_ID,
    APPLICATION_ORDER_BY,
    DESTINATION_COLUMN_ID,
    DESTINATION_ORDER_BY,
    ENDPOINT_TYPE,
    IMG_SRC,
    STATUS_COLUMN_ID,
    STATUS_ORDER_BY,
    TIME_COLUMN_ID,
    TIME_ORDER_BY,
    VERSION_COLUMN_ID,
    VERSION_ORDER_BY,
} from '../constants';
import { deploymentStateMock } from '../__mocks__/external-deployments.mocks';

describe('utils', () => {
    describe('getDeepLink', () => {
        it('should combine endpointUrl with applicationPath', () => {
            expect(getDeepLink('endp', 'oint')).toStrictEqual('endpoint');
            expect(getDeepLink('endp', undefined)).toStrictEqual('endp');
        });
    });

    describe('getServerIcon', () => {
        it('should give icon for type, default is empty', () => {
            expect(getServerIcon(ENDPOINT_TYPE.deploy)).toStrictEqual(IMG_SRC.deploy);
            expect(getServerIcon(ENDPOINT_TYPE.argoCd)).toStrictEqual(IMG_SRC.argoCd);
            expect(getServerIcon('')).toStrictEqual('');
        });
    });

    describe('mapOrderByValue', () => {
        it('should map columnName to correct orderBy value', () => {
            expect(mapOrderByValue(APPLICATION_COLUMN_ID)).toStrictEqual(APPLICATION_ORDER_BY);
            expect(mapOrderByValue(DESTINATION_COLUMN_ID)).toStrictEqual(DESTINATION_ORDER_BY);
            expect(mapOrderByValue(STATUS_COLUMN_ID)).toStrictEqual(STATUS_ORDER_BY);
            expect(mapOrderByValue(VERSION_COLUMN_ID)).toStrictEqual(VERSION_ORDER_BY);
            expect(mapOrderByValue(TIME_COLUMN_ID)).toStrictEqual(TIME_ORDER_BY);
        });
        it('should map unknown columnName to default orderBy of STATE', () => {
            expect(mapOrderByValue('other')).toStrictEqual(STATUS_ORDER_BY);
        });
    });

    describe('getTrimmedCommitHashFromCommitUrl', () => {
        it('should get trimmed commit hash from commit url', () => {
            expect(
                getTrimmedCommitHashFromCommitUrl('https://github.com/mkeretic1/argocd-sandbox/commit/d654f0c68f9213b19818ba23d91350886362431a'),
            ).toStrictEqual('d654f0c');
            expect(getTrimmedCommitHashFromCommitUrl('https://github.com/mkeretic1/argocd-sandbox/commit/d654f0')).toStrictEqual('d654f0');
        });
    });

    describe('getCommitHashFromCommitUrl', () => {
        it('should get commit hash from commit url', () => {
            expect(getCommitHashFromCommitUrl('https://github.com/mkeretic1/argocd-sandbox/commit/d654f0c68f9213b19818ba23d91350886362431a')).toStrictEqual(
                'd654f0c68f9213b19818ba23d91350886362431a',
            );
        });
    });

    describe('getVersionLabel', () => {
        it('should trim ArgoCD version label', () => {
            expect(
                getVersionLabel(ENDPOINT_TYPE.argoCd, 'https://github.com/mkeretic1/argocd-sandbox/commit/d654f0c68f9213b19818ba23d91350886362431a'),
            ).toStrictEqual('#d654f0c');
        });
        it('should not trim Deploy version label', () => {
            expect(getVersionLabel(ENDPOINT_TYPE.deploy, '1.0')).toStrictEqual('1.0');
        });
    });

    describe('getVersionTooltip', () => {
        it('should return commit hash from commit url for ArgoCD', () => {
            expect(
                getVersionTooltip(ENDPOINT_TYPE.argoCd, 'https://github.com/mkeretic1/argocd-sandbox/commit/d654f0c68f9213b19818ba23d91350886362431a'),
            ).toStrictEqual('d654f0c68f9213b19818ba23d91350886362431a');
        });
        it('should return version label for Deploy', () => {
            expect(getVersionTooltip(ENDPOINT_TYPE.deploy, '1.0')).toStrictEqual('1.0');
        });
    });

    describe('getVersionLink', () => {
        it('should return commit url for ArgoCD', () => {
            expect(
                getVersionLink(
                    ENDPOINT_TYPE.argoCd,
                    'https://github.com/mkeretic1/argocd-sandbox/commit/d654f0c68f9213b19818ba23d91350886362431a',
                    'http://deploy',
                    '/app1',
                ),
            ).toStrictEqual('https://github.com/mkeretic1/argocd-sandbox/commit/d654f0c68f9213b19818ba23d91350886362431a');
        });
        it('should return deep link for Deploy', () => {
            expect(getVersionLink(ENDPOINT_TYPE.deploy, '1.0', 'http://deploy', '/app1')).toStrictEqual('http://deploy/app1');
        });
    });

    describe('getDiscoveryCardSubheader', () => {
        it('should give subheader for type, default is empty', () => {
            expect(getDiscoveryCardSubheader(ENDPOINT_TYPE.deploy)).toStrictEqual('by Digital.ai');
            expect(getDiscoveryCardSubheader(ENDPOINT_TYPE.argoCd)).toStrictEqual('by Argo');
            expect(getDiscoveryCardSubheader('')).toStrictEqual('');
        });
    });

    describe('getDiscoveryCardTitle', () => {
        it('should give title for type, default is empty', () => {
            expect(getDiscoveryCardTitle(ENDPOINT_TYPE.deploy)).toStrictEqual('Deploy');
            expect(getDiscoveryCardTitle(ENDPOINT_TYPE.argoCd)).toStrictEqual('ArgoCD');
            expect(getDiscoveryCardTitle('')).toStrictEqual('');
        });
    });

    describe('getFilterCardTitle', () => {
        it('should give title for type, default is empty', () => {
            expect(getFilterCardTitle(ENDPOINT_TYPE.deploy)).toStrictEqual('Deploy Dev instance');
            expect(getFilterCardTitle(ENDPOINT_TYPE.argoCd)).toStrictEqual('ArgoCD instance');
            expect(getFilterCardTitle('')).toStrictEqual('');
        });
    });

    describe('generateEnvironmentTitle', () => {
        it('should generate environment title for state and type, default is empty', () => {
            expect(generateEnvironmentTitle(deploymentStateMock, ENDPOINT_TYPE.deploy)).toStrictEqual('Environment');
            expect(generateEnvironmentTitle(deploymentStateMock, ENDPOINT_TYPE.argoCd)).toStrictEqual('Environment / namespace');
            expect(generateEnvironmentTitle(deploymentStateMock, '')).toStrictEqual('');
        });
    });
});
