import React, { memo, useState, useEffect } from 'react';
import { DotAvatar, DotButton, DotCard, DotCardContent, DotIcon, DotIconButton, DotMenu, DotPill, MenuItemProps } from '@digital-ai/dot-components';
import moment from 'moment';
import upperFirst from 'lodash/upperFirst';
import { ManagedApplicationType } from '../../managed-application.types';
import './managed-application.component.less';
import { PluginInfo } from '../../../external-deployments/components/wizard/basic-wizard/application-discovery-basic.component';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getValidServerCards } from '../../ducks/managed-application.selectors';
import { STATUS_HTTP_CONNECTION } from '../../../external-deployments/constants';
import { folderExternalDeployments } from '../../../external-deployments/ducks/external-deployments.reducer';
const { getAvailablePlugins } = folderExternalDeployments.actions;

const ManagedApplicationComponent = ({
    applicationName,
    managedBy,
    dateCreated,
    applicationReference: targetDeployment,
    environmentTag,
    connectionServer,
}: ManagedApplicationType) => {
    const isManagedByRelease = managedBy === 'Release';
    const { name: targetConnection, type: serverType } = connectionServer;
    const validServerCards: PluginInfo[] = useAppSelector(getValidServerCards);

    const dispatch = useAppDispatch();

    useEffect(() => {
        dispatch(getAvailablePlugins(STATUS_HTTP_CONNECTION));
    }, []);

    const serverIcon = validServerCards.find((validServerCard) => validServerCard.type === serverType)?.iconLocation || '';
    const [open, setOpen] = useState(false);
    const [anchorEl, setAnchorEl] = useState<undefined | HTMLElement>(undefined);
    const handleToggle = (event: React.MouseEvent<HTMLButtonElement, MouseEvent>) => {
        setAnchorEl(event.currentTarget);
        setOpen(!open);
    };
    const menuItems: MenuItemProps[] = [
        { key: 'edit', children: 'Edit' },
        { key: 'delete', children: 'Delete' },
    ];

    return (
        <DotCard className="created-application-card" key={applicationName}>
            <div data-testid={`app${applicationName}`}>
                <DotCardContent>
                    <div className="application-name-column" title={applicationName}>
                        {applicationName}
                    </div>

                    <div className="managed-by-column">
                        <DotPill
                            backgroundcolor={isManagedByRelease ? '#E8EDF3 !important' : '#F3F5F6 !important'}
                            bordercolor={isManagedByRelease ? '#3D6C9E' : undefined}
                            icon={<DotIcon iconId={isManagedByRelease ? 'release' : 'lock'} />}
                            label={upperFirst(managedBy)}
                            labelcolor="#3B485C !important"
                            size="small"
                            status="success"
                            variant="outlined"
                        />
                    </div>

                    <div className="date-created-column">{moment(dateCreated).format('MMM DD, YYYY')}</div>

                    <div className="target-deployment-column">
                        <DotAvatar alt="ArgoCD Logo" imageSrc={serverIcon} size="small" />
                        {targetDeployment}
                    </div>

                    <div className="environment-tag-column">
                        <DotPill
                            backgroundcolor="#E9F0E0 !important"
                            bordercolor="#498500"
                            icon={<DotIcon iconId="environment" />}
                            label={environmentTag}
                            labelcolor="#3B485C !important"
                            size="small"
                            status="success"
                            variant="outlined"
                        />
                    </div>

                    <div className="target-connection-column">{targetConnection}</div>

                    <div className="link-column">
                        <DotButton tooltip="View deployments" type="outlined">
                            View deployments
                        </DotButton>
                    </div>

                    <div className="actions-column">
                        <div className="actions">
                            <DotIconButton iconId="options" onClick={handleToggle} />
                        </div>
                        <DotMenu
                            anchorEl={anchorEl}
                            disablePortal={true}
                            id="actions-menu"
                            menuItems={menuItems}
                            onLeave={() => setOpen(false)}
                            onSelect={() => setOpen(false)}
                            open={open}
                        />
                    </div>
                </DotCardContent>
            </div>
        </DotCard>
    );
};

export const ManagedApplication = memo(ManagedApplicationComponent);
