import React, { useEffect, useState, memo } from 'react';
import { useDispatch } from 'react-redux';
import noop from 'lodash/noop';
import { DotForm, DotTypography, DotButton, DotAutoComplete, AutoCompleteOption, DotCardHeader, DotCard, DotIconButton } from '@digital-ai/dot-components';
import './filter-applications-component.less';
import { Server, WebhookSource, WebhookSourceFilter } from '../../../external-deployment.types';
import { folderExternalDeployments } from '../../../ducks/external-deployments.reducer';
import { getWebhookSourceFiltersSelector } from '../../../ducks/external-deployments.selectors';
import { useAppSelector } from '../../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { IMG_SRC } from '../../../constants';
import { EventSourceForm } from '../event-source/application-event-source.component';
import { PatchModal } from '../patch-modal/patch-modal-component';
import { applicationManagement } from '../../../../applications-management/ducks/managed-application.reducer';
import { PluginInfo } from '../basic-wizard/application-discovery-basic.component';

export interface FilterApplicationsProps {
    createManagedApplications: boolean;
    onCancel: () => void;
    onFilterBack: () => void;
    onSave: () => void;
    plugins: PluginInfo[] | undefined;
    server: Server | undefined;
    webhookSource: WebhookSource;
}

const { loadWebhookSourceFilters, updateWebhookSourceFilters, patchExternalDeployments } = folderExternalDeployments.actions;
const { createManagedApplicationsFromWebhookSource } = applicationManagement.actions;

const FilterApplicationsComponent = (props: FilterApplicationsProps) => {
    const { onCancel, onFilterBack, onSave, server, webhookSource, createManagedApplications, plugins } = props;

    const webhookSourceFilters: WebhookSourceFilter | undefined = useAppSelector(getWebhookSourceFiltersSelector);

    const [isWebhookSourceForm, setIsWebhookSourceForm] = useState<boolean>(false);
    const [folderOptions, setFolderOptions] = useState<Array<string>>([]);
    const [selectedFolders, setSelectedFolders] = useState<Array<string>>([]);
    const [isModalOpen, setIsModalOpen] = useState<boolean>(false);

    const dispatch = useDispatch();
    const webhookSourceId = webhookSource.id || '';
    const serverType = server?.type || '';
    const plugin = plugins?.find((pluginInfo) => pluginInfo.type === serverType);
    const isAutoconfigurable = plugin ? plugin.isAutoconfigurable : false;
    const title = plugin ? plugin.title : '';
    const icon = plugin ? plugin.iconLocation : '';
    const eventSourceId = webhookSource.eventSource || '';

    const onFilterSave = () => {
        if (createManagedApplications) dispatch(createManagedApplicationsFromWebhookSource(webhookSourceId));
        onSave();
    };

    const saveApplicationFilters = () => {
        dispatch(
            updateWebhookSourceFilters({
                webhookSourceId,
                filteredFolders: selectedFolders,
            }),
        );
        if (isAutoconfigurable) {
            setIsModalOpen(true);
        } else {
            onFilterSave();
        }
    };

    const handleSelectedFoldersChange = (v: Array<AutoCompleteOption>) => {
        setSelectedFolders(v.map((value) => value.title));
    };

    const onEventSourceFormClose = () => {
        setIsWebhookSourceForm(false);
    };

    const onModalUpdateAutomatically = () => {
        dispatch(patchExternalDeployments(webhookSourceId));
        setIsModalOpen(false);
        onFilterSave();
    };

    const onModalUpdateManually = () => {
        setIsModalOpen(false);
        onFilterSave();
    };

    useEffect(() => {
        dispatch(loadWebhookSourceFilters(webhookSourceId));
    }, []);

    useEffect(() => {
        setFolderOptions(webhookSourceFilters ? webhookSourceFilters.folderFilterOptions : []);
        setSelectedFolders(webhookSourceFilters ? webhookSourceFilters.folderFilterValues : []);
    }, [webhookSourceFilters]);

    const generateApplicationFilters = () => {
        return (
            <div className="filter-applications">
                <DotForm onSubmit={noop}>
                    <DotTypography variant="h1">Map folders</DotTypography>
                    <DotAutoComplete
                        className="folders-filter"
                        freesolo={false}
                        helperText={'Map folders to display applications from'}
                        inputId="webhook-source-folders-filter"
                        label="Choose folders"
                        onChange={(e, v) => handleSelectedFoldersChange(v as Array<AutoCompleteOption>)}
                        options={folderOptions.map((option) => ({ id: option, title: option }))}
                        value={selectedFolders.map((option) => ({ id: option, title: option }))}
                    />
                    <DotCard className="filter-card">
                        <DotCardHeader
                            avatar={<img alt={serverType} src={icon} />}
                            className="card-header-inline"
                            subheader={server?.url}
                            subheaderSize="small"
                            title={title}
                            titleSize="small"
                        />
                    </DotCard>
                    <DotCard className="filter-card">
                        <DotCardHeader
                            action={
                                <DotIconButton
                                    color="inherit"
                                    data-testid="filter-webhook-edit-action"
                                    iconId="edit"
                                    iconSize="small"
                                    onClick={() => setIsWebhookSourceForm(true)}
                                    size="medium"
                                    tooltip="Edit webhook endpoint"
                                />
                            }
                            avatar={<img alt={webhookSource.title} src={IMG_SRC.webhookSource} />}
                            className="card-header-inline"
                            title={webhookSource.title}
                            titleSize="small"
                        />
                    </DotCard>
                    <div>
                        <DotButton className="button cancel" onClick={onFilterBack} type="text">
                            Back
                        </DotButton>
                        <span className="filter-end-buttons">
                            <DotButton className="button cancel" onClick={onCancel} type="outlined">
                                Cancel
                            </DotButton>
                            <DotButton
                                className="button primary"
                                data-testid="filter-apps-done-btn"
                                disabled={selectedFolders.length === 0}
                                onClick={saveApplicationFilters}
                            >
                                Done
                            </DotButton>
                        </span>
                    </div>
                </DotForm>
                <PatchModal
                    isModalOpen={isModalOpen && !isWebhookSourceForm}
                    onModalUpdateAutomatically={onModalUpdateAutomatically}
                    onModalUpdateManually={onModalUpdateManually}
                />
            </div>
        );
    };

    const generateWebhookSourceForm = () => {
        return <EventSourceForm closeForm={onEventSourceFormClose} eventSourceId={eventSourceId} />;
    };

    return (
        <>
            {!isWebhookSourceForm && generateApplicationFilters()}
            {isWebhookSourceForm && generateWebhookSourceForm()}
        </>
    );
};

export const FilterApplications = memo(FilterApplicationsComponent);
