import { SagaIterator } from 'redux-saga';
import { all, call, put, race, select, take, takeEvery, takeLatest } from 'redux-saga/effects';
import { applicationManagement } from './managed-application.reducer';
import {
    managedApplicationSaga,
    executeFetchManagedApplicationsAction,
    getFiltersAction,
    storeFiltersAction,
    executeCreateManagedApplicationsFromWebhookSourceAction,
    toaster,
    withLoadingState,
    navigateToWorkflowSaga,
    managedApplicationIdVariableKey,
} from './managed-application.saga';

import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { getApplicationManagementState } from './managed-application.selectors';
import { httpDELETE, httpGET, httpPOST } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { getReleaseVariables, updateReleaseVariable } from '../../../../../../../../../core/xlr-ui/app/features/tasks/ducks/variables-service.saga';
import { workflow } from '../../../../../../../../../core/xlr-ui/app/features/workflow/ducks/workflow.reducer';
import { ManagedApplicationsPage } from '../managed-application.types';
import { DELETE_BUTTON_ACTION } from '../components/created-applications-table/managed-application.component';

const {
    createManagedApplicationsFromWebhookSource,
    getFilters,
    loadManagedApplications,
    navigateToWorkflow,
    setCondition,
    setManagedApplications,
    setManagedApplicationsCount,
    setManagedApplicationsTotalCount,
    setPage,
    storeFilters,
} = applicationManagement.actions;
const { createWorkflowError, createWorkflowSuccess, runWorkflow } = workflow.actions;

describe('applicationManagementSaga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = managedApplicationSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeEvery(createManagedApplicationsFromWebhookSource, executeCreateManagedApplicationsFromWebhookSourceAction),
                takeEvery(getFilters, getFiltersAction),
                takeLatest(loadManagedApplications, executeFetchManagedApplicationsAction),
                takeLatest(navigateToWorkflow, navigateToWorkflowSaga),
                takeEvery(storeFilters, storeFiltersAction),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    const condition = 'con';
    const page: ManagedApplicationsPage = {
        folderId: 'folder-1',
        order: 'asc',
        orderBy: 'environment',
        page: 0,
        resultsPerPage: 10,
    };
    const filters = {
        condition,
        ...page,
    };
    const setManagedApplicationsFilters = jest.fn();
    const getManagedApplicationsFilters = jest.fn(() => filters);
    const clientSettings = {
        getManagedApplicationsFilters,
        setManagedApplicationsFilters,
    };
    const update = jest.fn();
    const filtersQueryParams = {
        update,
    };

    describe('storeFiltersAction', () => {
        it('should store filters to local storage', () => {
            const gen: SagaIterator = storeFiltersAction();
            expect(gen.next().value).toStrictEqual(select(getApplicationManagementState));
            expect(gen.next({ page, condition }).value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next(clientSettings).value).toStrictEqual(call(getAngularService, 'FiltersQueryParams'));
            expect(gen.next(filtersQueryParams).done).toBeTruthy();
            expect(update).toHaveBeenCalled();
        });
    });

    describe('executeCreateManagedApplicationsFromWebhookSourceAction', () => {
        it('should create managed applications from webhook source', () => {
            const webhookSourceId = 'Configuration/Custom/Configuration1682004056084295907f2e9c654f049b';

            const gen: SagaIterator = executeCreateManagedApplicationsFromWebhookSourceAction({ payload: webhookSourceId, type: 'any' });
            expect(gen.next().value).toEqual(
                call(withLoadingState, call(httpPOST, `/api/v1/managed-application/create-from-webhook-source/${webhookSourceId}`)),
            );
            expect(gen.next().value).toEqual(call(toaster.success, `Successfully created managed applications from discovered applications`));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeFetchManagedApplicationsAction', () => {
        it('should fetch managed apps', () => {
            const gen: SagaIterator = executeFetchManagedApplicationsAction();
            expect(gen.next().value).toStrictEqual(select(getApplicationManagementState));
            expect(gen.next({ page, condition }).value).toStrictEqual(
                call(
                    withLoadingState,
                    call(
                        httpGET,
                        `api/v1/managed-application?page=${page.page}&folderId=${page.folderId}&resultsPerPage=${
                            page.resultsPerPage
                        }&condition=${encodeURIComponent(condition)}`,
                        true,
                    ),
                ),
            );
            expect(gen.next({ data: { count: 2, managedApplications: [], totalCount: 200 } }).value).toStrictEqual(put(setManagedApplications([])));
            expect(gen.next().value).toStrictEqual(put(setManagedApplicationsCount(2)));
            expect(gen.next().value).toStrictEqual(put(setManagedApplicationsTotalCount(200)));
            expect(gen.next().done).toBeTruthy();
        });
        it('should handle undefined condition', () => {
            const gen: SagaIterator = executeFetchManagedApplicationsAction();
            expect(gen.next().value).toStrictEqual(select(getApplicationManagementState));
            expect(gen.next({ page }).value).toStrictEqual(
                call(
                    withLoadingState,
                    call(
                        httpGET,
                        `api/v1/managed-application?page=${page.page}&folderId=${page.folderId}&resultsPerPage=${page.resultsPerPage}&condition=`,
                        true,
                    ),
                ),
            );
            expect(gen.next({ data: { count: 2, managedApplications: [], totalCount: 200 } }).value).toStrictEqual(put(setManagedApplications([])));
            expect(gen.next().value).toStrictEqual(put(setManagedApplicationsCount(2)));
            expect(gen.next().value).toStrictEqual(put(setManagedApplicationsTotalCount(200)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('getFiltersAction', () => {
        it('should get filters from local storage and update state', () => {
            const gen: SagaIterator = getFiltersAction({ payload: 'folder-1', type: 'any' });
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next(clientSettings).value).toStrictEqual(call(getAngularService, '$stateParams'));
            expect(gen.next(clientSettings).value).toStrictEqual(put(setCondition(condition)));
            expect(gen.next(clientSettings).value).toStrictEqual(put(setPage(page)));
            expect(gen.next(clientSettings).done).toBeTruthy();
            expect(getManagedApplicationsFilters).toHaveBeenCalled();
        });
        it('should handle empty filters', () => {
            const getEmptyManagedApplicationsFilters = jest.fn();
            const gen: SagaIterator = getFiltersAction({ payload: 'folder-1', type: 'any' });
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next({ ...clientSettings, getManagedApplicationsFilters: getEmptyManagedApplicationsFilters }).value).toStrictEqual(
                call(getAngularService, '$stateParams'),
            );
            expect(gen.next({ getManagedApplicationsFilters: getEmptyManagedApplicationsFilters }).done).toBeTruthy();
            expect(getEmptyManagedApplicationsFilters).toHaveBeenCalled();
        });
    });

    describe('navigateToWorkflowSaga', () => {
        const appIdVariable = {
            id: 'varId',
            key: managedApplicationIdVariableKey,
        };
        it('should fetch managed app then trigger update workflow with updated variable and cleanup', () => {
            const payload = {
                actionName: 'update',
                managedApplicationId: 'appId',
                workflowId: 'id',
            };
            const action = { payload, type: 'any' };
            const gen: SagaIterator = navigateToWorkflowSaga(action);
            expect(gen.next().value).toStrictEqual(select(getApplicationManagementState));
            expect(gen.next({ page }).value).toStrictEqual(call(getReleaseVariables, payload.workflowId, true));
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            expect(gen.next([appIdVariable]).value).toStrictEqual(call(updateReleaseVariable, { ...appIdVariable, value: payload.managedApplicationId }));
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            expect(gen.next().value).toStrictEqual(put(runWorkflow({ folderId: page.folderId, workflow: { id: payload.workflowId } })));
            expect(gen.next().value).toStrictEqual(
                race({
                    error: take(createWorkflowError.type),
                    success: take(createWorkflowSuccess.type),
                }),
            );
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            expect(gen.next().value).toStrictEqual(call(updateReleaseVariable, { ...appIdVariable, value: '' }));
            expect(gen.next().done).toBeTruthy();
        });

        it('should handle null workflowId delete case', () => {
            const payload = {
                actionName: DELETE_BUTTON_ACTION,
                managedApplicationId: 'appId',
            };
            const action = { payload, type: 'any' };
            const gen: SagaIterator = navigateToWorkflowSaga(action);
            expect(gen.next().value).toStrictEqual(
                call(
                    toaster.info,
                    `Selected managed application has no connected ${payload.actionName} workflow,
            this action triggered deletion of Managed Application only in the Digital.a Release system`,
                ),
            );

            expect(gen.next().value).toStrictEqual(call(httpDELETE, `api/v1/managed-application/${payload.managedApplicationId}`));
            expect(gen.next().value).toStrictEqual(put(loadManagedApplications()));
            expect(gen.next().done).toBeTruthy();
        });

        it('should handle null workflowId update case', () => {
            const payload = {
                actionName: 'update',
                managedApplicationId: 'appId',
            };
            const action = { payload, type: 'any' };
            const gen: SagaIterator = navigateToWorkflowSaga(action);
            expect(gen.next().value).toStrictEqual(call(toaster.error, `Selected managed application has no connected update workflow`));

            expect(gen.next().done).toBeTruthy();
        });

        it('should handle create variable case', () => {
            const payload = {
                actionName: 'update',
                managedApplicationId: 'appId',
                workflowId: 'id',
            };
            const action = { payload, type: 'any' };
            const gen: SagaIterator = navigateToWorkflowSaga(action);
            expect(gen.next().value).toStrictEqual(select(getApplicationManagementState));
            expect(gen.next({ page }).value).toStrictEqual(call(getReleaseVariables, payload.workflowId, true));
            expect(gen.next([]).value).toStrictEqual(
                call(httpPOST, `api/v1/releases/Applications/${payload.workflowId}/variables`, {
                    id: null,
                    key: managedApplicationIdVariableKey,
                    requiresValue: false,
                    showOnReleaseStart: false,
                    type: 'xlrelease.StringVariable',
                    value: payload.managedApplicationId,
                }),
            );
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            expect(gen.next({ data: appIdVariable }).value).toStrictEqual(put(runWorkflow({ folderId: page.folderId, workflow: { id: payload.workflowId } })));
            expect(gen.next().value).toStrictEqual(
                race({
                    error: take(createWorkflowError.type),
                    success: take(createWorkflowSuccess.type),
                }),
            );
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            expect(gen.next().value).toStrictEqual(call(updateReleaseVariable, { ...appIdVariable, value: '' }));
            expect(gen.next().done).toBeTruthy();
        });
    });
});
