package com.xebialabs.xlrelease.status.service

import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.xlrelease.api.internal.views.{ExternalDeploymentOrderDirection, ExternalDeploymentOrderMode}
import com.xebialabs.xlrelease.api.v1.ConfigurationApi
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence
import com.xebialabs.xlrelease.service.ConfigurationAutoconfigService
import com.xebialabs.xlrelease.status.repository.persistence.ExternalDeploymentPersistence
import com.xebialabs.xlrelease.status.service.script.{ExternalDeploymentScriptService, FiltersRequestScriptService, PatchScriptService, formatErrorMessage}
import com.xebialabs.xlrelease.status.webhook.configuration.StatusHttpConnection
import com.xebialabs.xlrelease.status.webhook.events.{StatusWebhookEventSource, UpdateStatusEvent}
import com.xebialabs.xlrelease.webhooks.endpoint.WebhookEndpoint.PostWebhookEndpoint
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import java.lang
import java.util.{List => JList}
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

trait EndpointExternalDeploymentService {
  def getExternalDeployment(endpointIds: Vector[String],
                            maxAge: Long = 0,
                            page: lang.Long,
                            resultsPerPage: lang.Long,
                            orderBy: ExternalDeploymentOrderMode,
                            direction: ExternalDeploymentOrderDirection,
                            condition: String = ""): EndpointExternalDeployment
  def getWebhookSourceFilters(webhookSourceId: String): WebhookSourceFilters
  def saveWebhookSourceFilters(webhookSourceId: String, filteredFolders: List[String]): StatusWebhookEventSource
  def deleteWebhookSource(webhookSourceId: String): Unit
  def patchExternalDeployments(webhookSourceId: String): Unit
}

@Service
class EndpointExternalDeploymentServiceImpl(
                                             configurationApi: ConfigurationApi,
                                             configurationRepository: ConfigurationRepository,
                                             configurationPersistence: ConfigurationPersistence,
                                             externalDeploymentPersistence: ExternalDeploymentPersistence,
                                             statusScriptService: ExternalDeploymentScriptService,
                                             filtersRequestService: FiltersRequestScriptService,
                                             autoconfigService: ConfigurationAutoconfigService,
                                             patchScriptService: PatchScriptService
                                  ) extends EndpointExternalDeploymentService with Logging {

  override def getExternalDeployment(
                                      endpointIds: Vector[String],
                                      maxAge: Long,
                                      page: lang.Long,
                                      resultsPerPage: lang.Long,
                                      orderBy: ExternalDeploymentOrderMode,
                                      direction: ExternalDeploymentOrderDirection,
                                      condition: String): EndpointExternalDeployment = {
    val endpointToConnectionMap = endpointIds.map(endpointId => {
      val statusWebhookEventSource = configurationRepository.read[StatusWebhookEventSource](endpointId)
      val connection = configurationRepository.read[StatusHttpConnection](statusWebhookEventSource.sourceServer.getId)

      val maybeError = if (!externalDeploymentPersistence.exists(endpointId, maxAge)) {
        fetchAndPersistExternalDeployment(endpointId, statusWebhookEventSource, connection) match {
          case Left(str) =>
            Some(str)
          case _ =>
            None
        }
      }
      else {
        None
      }
      configurationPersistence.getUid(endpointId).get ->
        ConnectionServerData(connection.getId, connection.getTitle, connection.getType.toString, connection.getUrl, maybeError)
    }).toMap

    EndpointExternalDeployment(
      externalDeploymentPersistence.count(endpointIds, condition),
      endpointToConnectionMap,
      externalDeploymentPersistence.findForEndpoints(endpointIds, resultsPerPage, resultsPerPage * page, orderBy, direction, condition)
    )
  }

  private def fetchAndPersistExternalDeployment(endpointId: String,
                                                statusWebhookEventSource: StatusWebhookEventSource,
                                                connection: StatusHttpConnection): Either[String, Unit]  = {
    Try(statusScriptService.executeScript(statusWebhookEventSource)) match {
      case Success(result: JList[UpdateStatusEvent]) =>
        Right(externalDeploymentPersistence.batchSaveOrUpdate(endpointId, result.asScala.toVector))
      case Success(errorObject) =>
        val msg = s"Illegal response in ${connection.getId} statusScript response."
        error(msg + s" Returned type was ${errorObject.getClass.getName}, expected list of states")
        Left(msg)
      case Failure(exception) =>
        val msg = s"Error fetching Application status on ${connection.getId} [${connection.getTitle}]: ${exception.getMessage}"
        warn(msg)
        Left(msg)
    }
  }

  override def getWebhookSourceFilters(webhookSourceId: String): WebhookSourceFilters = {
    val statusWebhookEventSource = configurationRepository.read[StatusWebhookEventSource](webhookSourceId)
    val webhookSourceFilters = Try(filtersRequestService.executeScript(statusWebhookEventSource)) match {
      case Success(result: JList[String]) =>
        WebhookSourceFilters(
          webhookSourceId,
          result.asScala.toList,
          statusWebhookEventSource.filteredFolders.asScala.toList
        )
      case Success(errorObject) =>
        val msg = s"Illegal response in ${statusWebhookEventSource.getId} filtersRequest script response." +
          s" Returned type was ${errorObject.getClass.getName}, expected list of states"
        error(msg)
        throw WebhookSourceFiltersError(msg)
      case Failure(exception) =>
        val msg = s"Error fetching Folder list on $webhookSourceId [${statusWebhookEventSource.getTitle}]: ${exception.getMessage}"
        warn(msg)
        throw WebhookSourceFiltersError(formatErrorMessage(msg))
    }
    webhookSourceFilters
  }

  override def saveWebhookSourceFilters(webhookSourceId: String, filteredFolders: List[String]): StatusWebhookEventSource = {
    val statusWebhookEventSource = configurationRepository.read[StatusWebhookEventSource](webhookSourceId)
    val connection = configurationRepository.read[StatusHttpConnection](statusWebhookEventSource.sourceServer.getId)
    statusWebhookEventSource.filteredFolders = filteredFolders.asJava
    configurationRepository.update(statusWebhookEventSource)

    Try(autoconfigService.autoconfigure(statusWebhookEventSource)) match {
      case Success(_) =>
        //do nothing
      case Failure(exception) =>
        val msg = s"Error updating filteredFolders in updateFiltersRequest script on $webhookSourceId " +
          s"[${statusWebhookEventSource.getTitle}]: ${exception.getMessage}"
        warn(msg)
        throw UpdateFiltersError(formatErrorMessage(msg))
    }

    fetchAndPersistExternalDeployment(webhookSourceId, statusWebhookEventSource, connection) match {
      case Left(str) =>
        throw EndpointExternalDeploymentError(str)
      case _ => // nothing
    }

    statusWebhookEventSource
  }

  @IsTransactional
  override def deleteWebhookSource(webhookSourceId: String): Unit = {
    val eventSource: StatusWebhookEventSource = configurationRepository.read(webhookSourceId)
    val httpEndpoint: PostWebhookEndpoint = eventSource.getProperty("eventSource")

    configurationApi.deleteConfiguration(webhookSourceId)
    configurationApi.deleteConfiguration(httpEndpoint.getId)
  }

  override def patchExternalDeployments(webhookSourceId: String): Unit = {
    val statusWebhookEventSource = configurationRepository.read[StatusWebhookEventSource](webhookSourceId)
    Try(patchScriptService.executeScript(statusWebhookEventSource)) match {
      case Success(_) =>
      //do nothing
      case Failure(exception) =>
        val msg = s"Error executing patch script on $webhookSourceId [${statusWebhookEventSource.getTitle}]: ${exception.getMessage}"
        warn(msg)
        throw PatchError(formatErrorMessage(msg))
    }
  }
}

case class EndpointUpdateStatusEvent(endpointId: String, event: UpdateStatusEvent)

case class EndpointExternalDeployment(count: Int,
                                      connectionServers: Map[Integer, ConnectionServerData],
                                      states: Vector[EndpointUpdateStatusEvent])

case class ConnectionServerData(connectionId: String, title: String, serverType: String, serverUrl: String, error: Option[String])

case class EndpointExternalDeploymentError(msg: String) extends DeployitException(msg)

case class WebhookSourceFilters(webhookSourceId: String,
                                folderFilterOptions: List[String],
                                folderFilterValues: List[String])

case class WebhookSourceFiltersError(msg: String) extends DeployitException(msg)

case class UpdateFiltersError(msg: String) extends DeployitException(msg)

case class PatchError(msg: String) extends DeployitException(msg)
