package com.xebialabs.xlrelease.status.service

import com.xebialabs.xlrelease.api.internal.views.{ExternalDeploymentOrderDirection, ExternalDeploymentOrderMode}
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.status.repository.persistence.{ExternalDeployment, ExternalDeploymentPersistence, ExternalDeploymentUid}
import com.xebialabs.xlrelease.status.sse.service.ServerSentEventsService
import com.xebialabs.xlrelease.status.webhook.events.{CreateStatusEvent, DeleteStatusEvent, ExternalDeploymentEvent, UpdateStatusEvent}
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import scala.util.{Failure, Success, Try}

trait ExternalDeploymentService {
  def batchSaveOrUpdate(folderId: CiId, connectionUrl: String, states: Vector[UpdateStatusEvent]): Unit

  def findOnFolder(folderId: String,
                   limit: Long, offset: Long,
                   orderBy: ExternalDeploymentOrderMode,
                   direction: ExternalDeploymentOrderDirection,
                   condition: String = ""): Vector[EndpointUpdateStatusEvent]

  def exists(folderId: CiId, maxAge: Long): Boolean

  def count(folderId: CiId, condition: String): Int

  def saveAndNotify(connectionUrl: String, uid: ExternalDeploymentUid, processedEvent: ExternalDeploymentEvent): Unit
}

@Service
class ExternalDeploymentServiceImpl(
                                     externalDeploymentPersistence: ExternalDeploymentPersistence,
                                     serverSentEventsService: ServerSentEventsService,
                                   ) extends ExternalDeploymentService with Logging {

  override def batchSaveOrUpdate(folderId: CiId, connectionUrl: String, states: Vector[UpdateStatusEvent]): Unit = {
    states.foreach(state =>
      Try(externalDeploymentPersistence.save(
        ExternalDeploymentUid(folderId, state.applicationUid, state.destinationUid), ExternalDeployment(connectionUrl, state))) match {
        case Success(_) =>
          // handle application source and deployment target here for future implementation
        case Failure(exception) =>
          warn(s"Couldn't save fetched external deployment for Endpoint [${folderId}] into database, skipping record: ${exception.getMessage}")
      }
    )
  }


  override def saveAndNotify(connectionUrl: String,
                             uid: ExternalDeploymentUid,
                             externalDeploymentEvent: ExternalDeploymentEvent): Unit = {
    serverSentEventsService.send(externalDeploymentEvent)
    externalDeploymentEvent match {
      case _: DeleteStatusEvent if Option(uid.applicationCorrelationUid).isDefined && Option(uid.destinationCorrelationUid).isDefined =>
        externalDeploymentPersistence.delete(uid);
      case _: DeleteStatusEvent =>
        // skip empty DeleteStatusEvent
      case createEvent: CreateStatusEvent =>
        debug(s"Received CreateStatusEvent [${createEvent.applicationPath} on ${createEvent.destination}]- skipping save to database.")
      case updateStatus: UpdateStatusEvent =>
        externalDeploymentPersistence.save(uid, ExternalDeployment(connectionUrl, updateStatus))
      case event => // all the ignored types for now
        warn(s"Couldn't execute ExternalDeploymentEvent[${event.getClass.getName}] persistence - skipping save to database.")
    }
  }

  override def findOnFolder(folderId: CiId,
                            limit: Long, offset: Long,
                            orderBy: ExternalDeploymentOrderMode,
                            direction: ExternalDeploymentOrderDirection,
                            condition: String): Vector[EndpointUpdateStatusEvent] =
    externalDeploymentPersistence.findForEndpoints(folderId, limit, offset, orderBy, direction, condition)

  override def exists(folderId: CiId, maxAge: Long): Boolean =
    externalDeploymentPersistence.exists(folderId, maxAge)

  override def count(folderId: CiId, condition: String): Int =
    externalDeploymentPersistence.count(folderId, condition)
}
