package com.xebialabs.xlrelease.status.service

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.ConfigurationApi
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.Configuration
import com.xebialabs.xlrelease.domain.environments.LiveDeploymentConfig
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.service.ConfigurationAutoconfigService
import com.xebialabs.xlrelease.status.webhook.configuration.StatusHttpConnection
import com.xebialabs.xlrelease.status.webhook.events.StatusWebhookEventSource
import com.xebialabs.xlrelease.views.AutoconfigResponse
import com.xebialabs.xlrelease.webhooks.authentication.NoAuthentication
import com.xebialabs.xlrelease.webhooks.endpoint.WebhookEndpoint.PostWebhookEndpoint
import org.springframework.stereotype.Service

import java.util.UUID

trait EndpointSetupDeploymentStatusService {
  def setupApplicationStatusWebhook(serverId: String, folderId: String): WebhookBasicSetup
}

@Service
class SetupApplicationStatusServiceImpl(
                                         configurationApi: ConfigurationApi,
                                         configurationAutoconfigService: ConfigurationAutoconfigService,
                                         configurationRepository: ConfigurationRepository,
                                       ) extends EndpointSetupDeploymentStatusService {

  @IsTransactional
  override def setupApplicationStatusWebhook(configId: String, folderId: String): WebhookBasicSetup = {
    val config = configurationRepository.read[LiveDeploymentConfig](configId)
    val server = config.getTargetServer.asInstanceOf[StatusHttpConnection]
    val webhookEndpoint: Configuration = createHttpEndpointForWebhook(folderId, server.getTitle)
    val statusWebhookSource: Configuration = createStatusWebhookSource(
      folderId,
      server.getTitle,
      webhookEndpoint,
      config
    )

    val autoconfigResponse: AutoconfigResponse = configurationAutoconfigService.autoconfigure(statusWebhookSource)
    WebhookBasicSetup(
      autoconfigResponse.getSuccess,
      autoconfigResponse.getMessage,
      autoconfigResponse.getDetails,
      autoconfigResponse.getUrl,
      statusWebhookSource.getId
    )
  }

  private def createHttpEndpointForWebhook(folderId: String, title: String): Configuration = {

    val webhookEndpoint: PostWebhookEndpoint = Type.valueOf(classOf[PostWebhookEndpoint])
      .getDescriptor.newInstance("post-webhook-endpoint")

    webhookEndpoint.setTitle(title)
    webhookEndpoint.setFolderId(folderId)
    webhookEndpoint.setProperty("path", s"path-${UUID.randomUUID}")
    webhookEndpoint.setProperty("authentication", new NoAuthentication)

    configurationApi.addConfiguration(webhookEndpoint)
  }

  private def createStatusWebhookSource(folderId: String,
                                        title: String,
                                        eventSource: Configuration,
                                        config: LiveDeploymentConfig): Configuration = {
    val statusWebhookEventSourceType: String = config.getTargetServer.asInstanceOf[StatusHttpConnection].getStatusWebhookEventSourceName()

    val statusWebhookEventSource: StatusWebhookEventSource = Type.valueOf(statusWebhookEventSourceType)
      .getDescriptor.newInstance("status-event-webhook")
    statusWebhookEventSource.setTitle(title)
    statusWebhookEventSource.setFolderId(folderId)
    statusWebhookEventSource.setProperty("eventSource", eventSource)
    statusWebhookEventSource.setProperty("config", config)

    configurationApi.addConfiguration(statusWebhookEventSource)
  }
}

case class WebhookBasicSetup(success: Boolean,
                             message: String,
                             details: String,
                             url: String,
                             webhookSourceId: String)
