import React from 'react';
import { DotTypography } from '@digital-ai/dot-components';
import { ReactWrapper } from 'enzyme';
import { mountWithStoreAndTheme } from '@xlr-ui/tests/unit/testing-utils';
import { ENDPOINT_TYPE } from '../../constants';
import { folderExternalDeployments, initialState } from '../../ducks/external-deployments.reducer';
import { ApplicationCreateServerFormProp, CreateServerForm } from './application-create-server-component';
import { taskDrawerConnection } from '@xlr-ui/app/features/tasks/ducks/task-drawer-connection.reducer';
import { ciPropertyMapMock, propertiesByNameMock } from '@xlr-ui/app/features/configuration/components/instance/__mocks__/configuration-instance-form.mock';
import * as angularAccessor from '@xlr-ui/app/features/common/services/angular-accessor';
import { ConfirmLeaveService } from '@xlr-ui/app/features/tasks/types/angular';
import { ConnectionDialog } from '@xlr-ui/app/features/tasks/components/rails/common/connection-dialog/connection-dialog.component';

const dispatch = jest.fn();
const { initializeConnection } = taskDrawerConnection.actions;
const { updateServerList } = folderExternalDeployments.actions;
const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance<ConfirmLeaveService, [name: unknown]>;

describe('Create server component', () => {
    let wrapper: ReactWrapper;
    const closeForm = jest.fn();
    const defaultProps = {
        closeForm,
        folder: { id: 'folderId', title: 'Folder-1' },
        serverType: ENDPOINT_TYPE.deploy,
    };

    const getConfigurationHeaderTypography = () => wrapper.findWhere((node) => node.is(DotTypography) && node.props().className === 'configuration-header');
    const getConnectionDialog = () => wrapper.find(ConnectionDialog);

    const mountComponent = (props: ApplicationCreateServerFormProp = defaultProps) => {
        const state = {
            folderExternalDeployments: initialState,
            validServerCards: [ENDPOINT_TYPE.deploy],
            initializeConnection,
            updateServerList,
            taskDrawerConnection: {
                connection: {
                    autoconfigureTooltipMessage: 'The auto-configuration is not enabled for this Configuration.',
                    descriptor: {
                        type: ENDPOINT_TYPE.deploy,
                        label: 'Digital.ai Deploy Server (Container)',
                        virtual: false,
                        root: 'Configuration' as never,
                        description: 'Configure Deploy Server.',
                        properties: [
                            {
                                name: 'folderId',
                                fqn: 'deploy.Server.folderId',
                                label: 'Folder Id',
                                kind: 'STRING',
                                description: 'If set, this field contains the folder that this configuration belongs to.',
                                category: 'FOLDER_CONFIG',
                                asContainment: false,
                                inspection: false,
                                required: false,
                                requiredInspection: false,
                                password: false,
                                transient: false,
                                readonly: false,
                                size: 'DEFAULT',
                                referencedType: null,
                                default: null,
                            },
                        ],
                    },
                },
            },
            configuration: {
                categories: ['Common'],
                ciPropertyMap: ciPropertyMapMock,
                ciPropertyNames: ['eventSource'],
                propertiesByCategory: {
                    FOLDER_CONFIG: [],
                    Common: ['title', 'url'],
                },
                lookupMap: {},
                propertiesByName: propertiesByNameMock,
                isVariableMappingAllowed: true,
                isFormValid: false,
                isFormDirty: false,
                configurationTest: {
                    errorText: null,
                    isConnectionCheckInProgress: false,
                    isScriptFound: false,
                    isTestSuccessful: false,
                    isTested: false,
                },
                autoConfiguration: {
                    details: null,
                    isConfigurationSuccessful: false,
                    isConfigured: false,
                    isConfiguring: false,
                    message: null,
                    url: null,
                },
            },
        };
        wrapper = mountWithStoreAndTheme(<CreateServerForm {...props} />, dispatch, state);
    };

    beforeEach(() => {
        getAngularServiceSpy.mockReturnValue({
            requireConfirmation: jest.fn(),
            disableConfirmation: jest.fn(),
        } as never);
        mountComponent();
    });

    afterEach(() => {
        wrapper.unmount();
    });

    it('should render properly', () => {
        expect(wrapper.find(CreateServerForm)).toExist();
        const configurationHeaderTypography = getConfigurationHeaderTypography();
        expect(configurationHeaderTypography).toExist();
        const configurationHeaderTypographyProps = configurationHeaderTypography.props();
        expect(configurationHeaderTypographyProps.children).toBe('New Digital.ai Deploy Server (Container)');
        expect(configurationHeaderTypographyProps.variant).toBe('h2');
        const titleLabel = wrapper.find('[data-test-id="title"] .text-input-label').at(0).props().children as React.ReactNode[];
        const urlLabel = wrapper.find('[data-test-id="url"] .text-input-label').at(0).props().children as React.ReactNode[];

        expect(titleLabel[0]).toStrictEqual('Title');
        expect(urlLabel[0]).toStrictEqual('Url');
    });

    it('should execute correct handler when onCancel has been invoked on the ConnectionDialog component', () => {
        const connectionDialog = getConnectionDialog();
        connectionDialog.invoke('onCancel')();
        expect(closeForm).toHaveBeenCalled();
    });

    it('should execute correct handler when onSave has been invoked on the ConnectionDialog component', () => {
        const connectionDialog = getConnectionDialog();
        connectionDialog.invoke('onSave')();
        expect(closeForm).toHaveBeenCalled();
    });

    it('should pass correct referencedType when serverType is not defined', () => {
        mountComponent({ ...defaultProps, serverType: undefined });
        expect(getConnectionDialog().props().referencedType).toBe('');
    });
});
