package com.xebialabs.xlrelease.status.service.script

import com.fasterxml.jackson.core.`type`.TypeReference
import com.xebialabs.xlrelease.domain.environments.LiveDeploymentConfig
import com.xebialabs.xlrelease.json.JsonUtils.objectMapper
import com.xebialabs.xlrelease.runner.impl.RunnerScriptService
import com.xebialabs.xlrelease.script.XlrScriptContext
import com.xebialabs.xlrelease.script.jython.JythonScriptService
import com.xebialabs.xlrelease.status.service.script.CleanupScriptExecutor.createCleanupScriptContext
import com.xebialabs.xlrelease.status.webhook.events.{RemoteStatusWebhookEventSource, StatusWebhookEventSource}
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.stereotype.Component

trait CleanupScriptService {
  @throws(classOf[Exception])
  def executeScript(eventSource: StatusWebhookEventSource, existingLiveDeploymentConfigs: Set[LiveDeploymentConfig]): AnyRef
}

@Component
class CleanupScriptExecutor(
                             val scriptService: JythonScriptService,
                             runnerScriptService: RunnerScriptService
                           ) extends ScriptExecution with CleanupScriptService with Logging {

  @Timed
  @throws(classOf[Exception])
  override def executeScript(connection: StatusWebhookEventSource, existingLiveDeploymentConfigs: Set[LiveDeploymentConfig]): AnyRef = {
    logger.debug(s"Executing cleanup script for StatusWebhookEventSource ${connection.getName}")
    if (connection.isInstanceOf[RemoteStatusWebhookEventSource]) {
      val inputParams = buildDefaultInputParams(connection)
      inputParams.put("existingLiveDeploymentConfigs", objectMapper.writeValueAsString(existingLiveDeploymentConfigs.map(ciAsMap)))
      runnerScriptService.executeScript[AnyRef](
        connection.cleanupScript,
        inputParams,
        new TypeReference[AnyRef] {}
      )
    } else {
      execute(createCleanupScriptContext(connection))
    }
  }
}

private class CleanupScriptContextBuilder(connection: StatusWebhookEventSource) extends BaseScriptContextBuilder(connection) {

  override val scriptProperty: String = CleanupScriptProperty

  override protected def doBuild(context: XlrScriptContext): Unit = {
    addConnectionServerAttributes(context)
    addConnectionAttributes(context)
    addScript(context)
  }
}

private object CleanupScriptExecutor {
  def createCleanupScriptContext(connection: StatusWebhookEventSource): XlrScriptContext = {
    new CleanupScriptContextBuilder(connection).build()
  }
}
