package com.xebialabs.xlrelease.plugins.dashboard.repository.sql.persistence

import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.{Sql, SqlBuilder}
import com.xebialabs.xlrelease.plugins.dashboard.repository.sql.persistence.DashboardSchema.DASHBOARDS
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.FOLDERS
import com.xebialabs.xlrelease.repository.sql.persistence.SecurableSqlBuilder
import com.xebialabs.xlrelease.security.PermissionChecker.GLOBAL_SECURITY_ALIAS


object DashboardSqlBuilder {
  val alias: String = "ds"
  val folderAlias: String = "f"

  val STMT_DASHBOARD_SELECT: String =
    s"""|SELECT
        |   $alias.${DASHBOARDS.ID},
        |   $folderAlias.${FOLDERS.FOLDER_ID},
        |   $folderAlias.${FOLDERS.FOLDER_PATH},
        |   $alias.${DASHBOARDS.CONTENT}
        | FROM ${DASHBOARDS.TABLE} $alias
        | LEFT JOIN ${FOLDERS.TABLE} $folderAlias ON $folderAlias.${FOLDERS.FOLDER_ID} = $alias.${DASHBOARDS.PARENT_ID}""".stripMargin
}

class DashboardSqlBuilder(implicit dialect: Dialect) extends SqlBuilder[DashboardSqlBuilder] with SecurableSqlBuilder {

  import DashboardSqlBuilder._

  def select(): DashboardSqlBuilder = {
    super.select(STMT_DASHBOARD_SELECT)
  }

  def withTitleLike(title: String): DashboardSqlBuilder = {
    like(s"$alias.${DASHBOARDS.TITLE}", title)
  }

  def withParent(parentId: CiId): DashboardSqlBuilder = {
    if (parentId != null && !parentId.isEmpty && parentId.toLowerCase() != GLOBAL_SECURITY_ALIAS) {
      conditions += Sql(s"$alias.${DASHBOARDS.PARENT_ID} = ?", Seq(getName(parentId)))
    } else {
      conditions += Sql(s"$alias.${DASHBOARDS.PARENT_ID} IS NULL", Seq.empty)
    }
    this
  }

  def withGlobalPermission(permission: Option[Permission], user: String, principals: Iterable[String], roleIds: Iterable[String]): DashboardSqlBuilder = {
    if (permission.isDefined) {
      conditions += Sql(
        s"""(
           |  LOWER($alias.${DASHBOARDS.OWNER}) = ?
           |  OR
           |  $alias.${DASHBOARDS.CI_UID} IN (
           |    ${selectCiIdsWithPermission(principals, roleIds)}
           |  )
           |)
      """.stripMargin,
        Seq(user.toLowerCase) ++ Seq(permission.get.getPermissionName) ++ principals.map(_.toLowerCase) ++ roleIds
      )
    }
    this
  }

  def withFolderPermission(permission: Option[Permission], user: String, principals: Iterable[String], roleIds: Iterable[String]): DashboardSqlBuilder = {
    if (permission.isDefined) {
      conditions += Sql(
        s"""(
           |  LOWER($alias.${DASHBOARDS.OWNER}) = ?
           |  OR
           |  $folderAlias.${FOLDERS.SECURITY_UID} IN (
           |    ${selectCiIdsWithPermission(principals, roleIds)}
           |  )
           |)
      """.stripMargin,
        Seq(user.toLowerCase) ++ Seq(permission.get.getPermissionName) ++ principals.map(_.toLowerCase) ++ roleIds
      )
    }
    this
  }

  def ordered(): DashboardSqlBuilder = {
    orderBy(s"$alias.${DASHBOARDS.CI_UID} ASC")
    this
  }

  override def newInstance: DashboardSqlBuilder = new DashboardSqlBuilder()

}
