import React, { useState } from 'react';
import moment from 'moment';
import { DotChip, DotColumnHeader, DotIcon, DotTable, DotTooltip, DotTypography } from '@digital-ai/dot-components';
import { ButtonWithLink } from '../../CommonCardTile/common-card-tile.component';
import { TemplateActionsButton } from './template-actions-button';
import { getReleaseCountText, getReleaseIdForUrl } from './helpers/helpers';
import { CommonActiveTile } from '../../CommonCardTile/common-active-tile.component';
import { useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getProfile } from '../../../../../../../../../../core/xlr-ui/app/features/profile/ducks/profile.selectors';
import { calculateDotDateFormat } from '../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/scheduling/helper';
import { RegionalSettings } from '../../../../../../../../../../core/xlr-ui/app/js/locale/regional-settings';
import './active-template-tile.component.less';

export interface HomeTemplate {
    folderId: string;
    folderName: string;
    folderPath: string;
    lastEditedByUser: number;
    releaseCount: number;
    releaseId: string;
    releaseTitle: string;
}

export interface ActiveTemplateTileProps {
    primaryButton: ButtonWithLink;
    secondaryButton?: ButtonWithLink;
    templates: Array<HomeTemplate>;
    title: string;
}

export const ActiveTemplateTile = ({ primaryButton, secondaryButton, templates, title }: ActiveTemplateTileProps) => {
    const [selectedEndIcon, setSelectedEndIcon] = useState<string>('');

    const { dateFormat: profileDateFormat } = useAppSelector(getProfile);
    const dateFormat = calculateDotDateFormat(RegionalSettings.getDateFormats(profileDateFormat).shortDate);

    const tableColumns: Array<DotColumnHeader> = [
        { id: 'templateName', label: 'Template', truncate: true },
        { id: 'lastTimeUpdated', label: 'Last updated', width: '180px' },
        { id: 'actionButton', label: '', width: '72px' },
    ];

    const templatesData = templates.map((template) => {
        const templateId = getReleaseIdForUrl(template);
        const releaseCountText = getReleaseCountText(template.releaseCount);

        return {
            id: templateId,
            rowData: {
                templateName: (
                    <>
                        <DotTypography data-testid="row-template-title" variant="body1">
                            {template.releaseTitle}
                        </DotTypography>
                        <DotTypography className="template-folder-name" variant="body2">
                            {template.folderName} ({releaseCountText})
                        </DotTypography>
                    </>
                ),
                lastTimeUpdated: (
                    <DotTooltip data-testid="last-time-updated-tooltip" title={`Last time updated ${moment(template.lastEditedByUser).format(dateFormat)}`}>
                        <DotChip size="small" startIcon={<DotIcon iconId="calendar" />}>
                            {moment(template.lastEditedByUser).fromNow()}
                        </DotChip>
                    </DotTooltip>
                ),
                actionButton: (
                    <TemplateActionsButton
                        onOptionsClick={(id) => setSelectedEndIcon(id)}
                        selected={selectedEndIcon === template.releaseId}
                        template={template}
                        urlTemplateId={templateId}
                    />
                ),
            },
        };
    });

    const onRowClick = (id: string) => {
        window.location.href = `#/templates/${id}`;
    };

    const mainContent = (
        <DotTable
            className="active-template-table"
            columns={tableColumns}
            data={templatesData}
            onRowClick={(_event, id) => onRowClick(id)}
            onUpdateData={() => templatesData}
            sortable={false}
        />
    );

    return (
        <div className="active-template-tile">
            <CommonActiveTile
                avatarIcon="template"
                content={mainContent}
                primaryButton={primaryButton}
                secondaryButton={secondaryButton}
                subtitle="Last updated - Most recent 5"
                title={title}
            />
        </div>
    );
};
