import React from 'react';
import { Dashboard, DashboardAction, DashboardActionMenuItemProps } from '../types';
import { checkIfDashboardHasEditPermission, createMenuItem, getDuplicatedDashboardFromOriginal, mapDashboardActions } from './index';
import { MenuItem } from '../../../../../../../../../../core/xlr-ui/app/react/components/menu-item/menu-item.component';

describe('global dashboard helper functions', () => {
    describe('createMenuItem', () => {
        it('should create a menu item with default values', () => {
            const props: DashboardActionMenuItemProps = {
                action: { actionId: '1', disabled: false },
                label: 'Default Item',
            };
            const menuItem = createMenuItem(props);
            expect(menuItem).toEqual({
                children: <MenuItem iconId={undefined} label="Default Item" />,
                disabled: false,
                key: '1',
                divider: false,
                items: undefined,
            });
        });

        it('should create a menu item with custom values', () => {
            const props: DashboardActionMenuItemProps = {
                action: { actionId: '2', disabled: true },
                label: 'Custom Item',
                iconId: 'custom-icon',
                divider: true,
                items: [{ label: 'Sub Item' }] as never,
            };
            const menuItem = createMenuItem(props);
            expect(menuItem).toEqual({
                children: <MenuItem iconId="custom-icon" label="Custom Item" />,
                disabled: true,
                key: '2',
                divider: true,
                items: [{ label: 'Sub Item' }],
            });
        });

        it('should create a menu item with no items', () => {
            const props: DashboardActionMenuItemProps = {
                action: { actionId: '3', disabled: false },
                label: 'No Items',
                items: [],
            };
            const menuItem = createMenuItem(props);
            expect(menuItem).toEqual({
                children: <MenuItem iconId={undefined} label="No Items" />,
                disabled: false,
                key: '3',
                divider: false,
                items: [],
            });
        });
    });

    describe('mapDashboardActions', () => {
        it('should map DUPLICATE action correctly', () => {
            const actions: DashboardAction[] = [{ actionId: 'DUPLICATE', disabled: false }];
            const mappedActions = mapDashboardActions(actions);
            expect(mappedActions).toEqual([
                {
                    children: <MenuItem iconId="duplicate" label="Duplicate" />,
                    disabled: false,
                    key: 'DUPLICATE',
                    divider: false,
                    items: undefined,
                },
            ]);
        });

        it('should map DELETE action correctly', () => {
            const actions: DashboardAction[] = [{ actionId: 'DELETE', disabled: true }];
            const mappedActions = mapDashboardActions(actions);
            expect(mappedActions).toEqual([
                {
                    children: <MenuItem iconId="delete" label="Delete" />,
                    disabled: true,
                    key: 'DELETE',
                    divider: false,
                    items: undefined,
                },
            ]);
        });

        it('should throw an error for an unknown action', () => {
            const actions: DashboardAction[] = [{ actionId: 'UNKNOWN_ACTION', disabled: false }];
            expect(() => mapDashboardActions(actions)).toThrowError('Wrong dashboard action [UNKNOWN_ACTION]');
        });

        it('should map multiple actions correctly', () => {
            const actions: DashboardAction[] = [
                { actionId: 'DUPLICATE', disabled: false },
                { actionId: 'DELETE', disabled: true },
            ];
            const mappedActions = mapDashboardActions(actions);
            expect(mappedActions).toEqual([
                {
                    children: <MenuItem iconId="duplicate" label="Duplicate" />,
                    disabled: false,
                    key: 'DUPLICATE',
                    divider: false,
                    items: undefined,
                },
                {
                    children: <MenuItem iconId="delete" label="Delete" />,
                    disabled: true,
                    key: 'DELETE',
                    divider: false,
                    items: undefined,
                },
            ]);
        });
    });

    describe('checkIfDashboardHasEditPermission', () => {
        const authenticator = {
            hasPermission: jest.fn(),
        };
        it('should return true if dashboard has edit permission', () => {
            const dashboardWithEditPermission: Dashboard = {
                description: 'Dashboard with edit permission',
                id: '123',
                metadata: {
                    security: {
                        permissions: ['dashboard#edit'],
                        teams: [],
                    },
                },
                owner: 'admin',
                parentId: null,
                templateType: 'default',
                title: 'Dashboard 1',
                tiles: [],
            };

            authenticator.hasPermission.mockReturnValueOnce(true);

            expect(checkIfDashboardHasEditPermission(authenticator as never, dashboardWithEditPermission)).toBe(true);
            expect(authenticator.hasPermission).toHaveBeenCalledWith('dashboard#edit', {
                security: {
                    permissions: ['dashboard#edit'],
                    teams: [],
                },
            });
        });

        it('should return false if dashboard does not have edit permission', () => {
            const dashboardWithoutEditPermission: Dashboard = {
                description: 'Dashboard without edit permission',
                id: '123',
                metadata: {
                    security: {
                        permissions: ['dashboard#view'],
                        teams: [],
                    },
                },
                owner: 'admin',
                parentId: null,
                templateType: 'default',
                title: 'Dashboard 2',
                tiles: [],
            };

            authenticator.hasPermission.mockReturnValueOnce(false);
            expect(checkIfDashboardHasEditPermission(authenticator as never, dashboardWithoutEditPermission)).toBe(false);
            expect(authenticator.hasPermission).toHaveBeenCalledWith('dashboard#edit', {
                security: {
                    permissions: ['dashboard#view'],
                    teams: [],
                },
            });
        });
    });

    describe('getDuplicatedDashboardFromOriginal', () => {
        const originalDashboard: Dashboard = {
            description: 'Original Description',
            id: 'originalId',
            metadata: { security: { permissions: [], teams: [] } },
            owner: 'admin',
            parentId: null,
            templateType: 'template',
            tiles: [
                { id: 'tile1', title: 'Tile 1', type: 'type1' },
                { id: 'tile2', title: 'Tile 2', type: 'type2' },
            ],
            title: 'Original Dashboard',
        };

        it('should duplicate the dashboard with a new title and nullify tile IDs', () => {
            const newTitle = 'Duplicated Dashboard';
            const duplicatedDashboard = getDuplicatedDashboardFromOriginal(originalDashboard, newTitle);

            expect(duplicatedDashboard.title).toBe(newTitle);
            expect(duplicatedDashboard.id).toBeNull();
            expect(duplicatedDashboard.metadata).toBeUndefined();

            duplicatedDashboard.tiles.forEach((tile) => {
                expect(tile.id).toBeNull();
            });
        });
    });
});
