package com.xebialabs.xlrelease.plugins.dashboard.service

import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, DescriptorRegistry, Type}
import com.xebialabs.xlrelease.domain.VariableContainer
import com.xebialabs.xlrelease.plugins.dashboard.domain.{Dashboard, Tile, TileScope}
import com.xebialabs.xlrelease.plugins.dashboard.repository.DelegatingDashboardRepository
import com.xebialabs.xlrelease.plugins.dashboard.service.TileHelper.resolveVariables
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.{getParentId, isFolderId, isReleaseId}
import com.xebialabs.xlrelease.security.PermissionChecker.GLOBAL_SECURITY_ALIAS
import com.xebialabs.xlrelease.service._
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils.replacePasswordPropertiesInCiIfNeeded
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class TileService @Autowired()(releaseService: ReleaseService,
                               folderService: FolderService,
                               variableService: VariableService,
                               dashboardRepository: DelegatingDashboardRepository,
                               tileScriptExecutor: TileScriptExecutor,
                               teamService: TeamService,
                               folderVariableService: FolderVariableService,
                               configurationVariableService: ConfigurationVariableService) extends Logging {

  val cacheBusterKey = "cb"
  val refreshKey = "refresh"

  @Timed
  @throws(classOf[Exception])
  def executeTileDataScript(tileId: String, additionalVariables: Map[String, Any]): Any = {
    val dashboardId = getParentId(tileId)
    val dashboard = dashboardRepository.findDashboardById(dashboardId)
    val tile = dashboard.getTile(tileId)
    val parent = getParent(dashboard)

    val globalVars = variableService.findGlobalVariablesOrEmpty()
    val globalVariables = globalVars.getVariables.asScala.toSeq
    val folderVariables = folderVariableService.getAllFromAncestry(Ids.findFolderId(dashboardId)).getVariables.asScala.toSeq

    val resolvedTile = resolveVariables(tile, parent, globalVariables ++ folderVariables)

    val variablesWithoutCacheBuster = additionalVariables - cacheBusterKey - refreshKey
    if (additionalVariables.contains(refreshKey)) {
      tileScriptExecutor.evictFromCache(parent, resolvedTile, variablesWithoutCacheBuster)
    }

    configurationVariableService.resolveFromCi(tile, globalVars)(_.getProperties.asScala)

    tileScriptExecutor.executeTileScript(parent, resolvedTile, variablesWithoutCacheBuster)
  }

  private def getParent(dashboard: Dashboard): Option[VariableContainer] = {
    Option(dashboard.getParentId).getOrElse(GLOBAL_SECURITY_ALIAS) match {
      case releaseId if isReleaseId(releaseId) =>
        val release = releaseService.findByIdIncludingArchived(releaseId)
        teamService.decorateWithEffectiveTeams(release)
        Some(release)
      case folderId if isFolderId(folderId) =>
        val folder = folderService.findById(folderId, 0)
        Some(folder)
      case _ => None
    }
  }

  @Timed
  def getTileTypes(scope: TileScope): Seq[Descriptor] = {
    DescriptorRegistry.getSubtypes(Type.valueOf(classOf[Tile]))
      .asScala
      .map(_.getDescriptor)
      .filter(desc => !desc.isVirtual && desc.newInstance[Tile]("dummy").isSupportedOn(scope))
      .toList
      .sortBy(_.getLabel)
  }

  @Timed
  def findTileById(tileId: String): Tile = dashboardRepository.findTileById(tileId)

  @Timed
  def updateTile(tile: Tile): Tile = {
    val original: Tile = findTileById(tile.getId)

    replacePasswordPropertiesInCiIfNeeded(Some(original), tile)

    dashboardRepository.updateTile(tile)
  }

}
