package com.xebialabs.xlrelease.plugins.dashboard.api.internal

import com.fasterxml.jackson.databind.ObjectMapper
import com.xebialabs.deployit.plugin.api.reflect.Descriptor
import com.xebialabs.xlrelease.plugins.dashboard.domain.TileScope
import com.xebialabs.xlrelease.plugins.dashboard.service.{DashboardSecurity, DashboardService}
import com.xebialabs.xlrelease.plugins.dashboard.views.DashboardView
import com.xebialabs.xlrelease.plugins.dashboard.views.DashboardView.toDashboard
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import java.util.{List => JList}
import jakarta.ws.rs._
import jakarta.ws.rs.core.MediaType
import scala.jdk.CollectionConverters._

@Path("/dashboards")
@Controller
@Produces(Array(MediaType.APPLICATION_JSON))
@Consumes(Array(MediaType.APPLICATION_JSON))
class DashboardResource @Autowired()(val dashboardService: DashboardService,
                                     val dashboardSecurity: DashboardSecurity,
                                     implicit val configurationRepository: ConfigurationRepository,
                                     implicit val objectMapper: ObjectMapper) {

  @GET
  def search(@QueryParam("parentId") parentId: String, @QueryParam("title") title: String): JList[DashboardView] = {
    //TODO check whether we need to do security check for parent ID
    dashboardService.search(parentId, title).map(DashboardView(_)).asJava
  }

  @GET
  @Path("{dashboardId:Applications.*|.*Dashboard[^/-]*}")
  def getDashboard(@PathParam("dashboardId") dashboardId: String, @QueryParam("refresh") refresh: Boolean): DashboardView = {
    dashboardSecurity.checkView(dashboardId)
    val dashboard = dashboardService.findDashboardById(dashboardId)
    if (refresh) {
      dashboardService.evictTilesCache(dashboard)
    }
    DashboardView(dashboard)
  }

  @PUT
  @Path("{dashboardId:Applications.*|.*Dashboard[^/-]*}")
  def updateDashboard(@PathParam("dashboardId") dashboardId: String, dashboard: DashboardView): DashboardView = {
    dashboardSecurity.checkEdit(dashboardId)
    dashboard.id = dashboardId
    val updatedDashboard = dashboardService.updateDashboard(toDashboard(dashboard))
    DashboardView(updatedDashboard)
  }

  @POST
  def createDashboard(dashboard: DashboardView): DashboardView = {
    dashboardSecurity.checkCreate(dashboard.parentId)
    dashboard.id = null
    val createdDashboard = dashboardService.createDashboard(toDashboard(dashboard))
    DashboardView(createdDashboard)
  }

  @DELETE
  @Path("{dashboardId:Applications.*|.*Dashboard[^/-]*}")
  def deleteDashboard(@PathParam("dashboardId") dashboardId: String): Unit = {
    dashboardSecurity.checkEdit(dashboardId)
    dashboardService.deleteDashboard(dashboardId)
  }

  @GET
  @Path("/templates")
  def listTemplates(@QueryParam("scope") @DefaultValue("RELEASE") scope: TileScope): JList[Descriptor] = {
    dashboardService.getDashboardTemplates(scope).asJava
  }
}
