package com.xebialabs.xlrelease.environments.repository.sql

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.v1.filter.ReservationFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.environments.{Environment, EnvironmentReservation}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.{ColumnAliases, EnvironmentsWithReservationsSqlBuilder}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.{EnvironmentPersistence, EnvironmentReservationPersistence}
import com.xebialabs.xlrelease.environments.repository.{ApplicationRepository, EnvironmentReservationRepository}
import com.xebialabs.xlrelease.repository.Page
import com.xebialabs.xlrelease.service.CiIdService
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Repository

import java.util.Date
import scala.jdk.CollectionConverters._

@IsTransactional
@Repository
class SqlEnvironmentReservationRepository @Autowired()(@Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                                       implicit val ciIdService: CiIdService,
                                                       applicationRepository: ApplicationRepository,
                                                       environmentPersistence: EnvironmentPersistence,
                                                       environmentReservationPersistence: EnvironmentReservationPersistence)
  extends EnvironmentReservationRepository {

  @IsReadOnly
  override def search(filters: ReservationFilters, page: Page): Map[Environment, Seq[EnvironmentReservation]] = {
    val query = EnvironmentsWithReservationsSqlBuilder()
      .orderBy(ColumnAliases.Environments.CI_UID)
      .limitAndOffset(page.resultsPerPage, page.offset)

    Option(filters).foreach { f =>
      Option(f.getEnvironmentTitle).foreach(query.withEnvironmentTitleLike)
      Option(f.getStages).foreach(stages => query.withStageTitles(stages.asScala))
      Option(f.getApplications).foreach(apps => query.withApplicationTitles(apps.asScala))
      Option(f.getLabels).foreach(labels => query.withLabelTitles(labels.asScala))
      Option(f.getFrom).foreach(query.from)
      Option(f.getTo).foreach(query.to)
    }

    environmentReservationPersistence.search(query.build())
  }

  @IsReadOnly
  override def findById(environmentReservationId: String): EnvironmentReservation = {
    environmentReservationPersistence.findById(environmentReservationId)
      .getOrElse(throw new NotFoundException(s"Environment reservation [$environmentReservationId] not found"))
  }

  @IsReadOnly
  override def getAllForEnvironmentId(environmentId: String): Seq[EnvironmentReservation] = {
    val tmpEnvironment = new Environment().getType.getDescriptor.newInstance[Environment](environmentId)
    val query = EnvironmentsWithReservationsSqlBuilder().withEnvironmentId(environmentId)
    environmentReservationPersistence.search(query.build())
      .getOrElse(tmpEnvironment, Seq.empty[EnvironmentReservation])
  }

  override def create(environmentReservation: EnvironmentReservation): EnvironmentReservation = {
    val resId = environmentReservationPersistence.insert(environmentReservation)
    findById(resId)
  }

  override def update(environmentReservation: EnvironmentReservation): EnvironmentReservation = {
    environmentReservationPersistence.update(environmentReservation)
    findById(environmentReservation.getId)
  }

  override def delete(environmentReservationId: String): Unit = {
    environmentReservationPersistence.delete(environmentReservationId)
  }

  override def existsByEnvironmentIdAndApplicationIdAndDate(environmentId: String, applicationId: String, date: Date): Boolean = {
    environmentReservationPersistence.exists(environmentId, applicationId, date)
  }

  override def findNearestComingByEnvironmentIdAndApplicationIdAndDate(environmentId: String, applicationId: String, date: Date): Option[Date] = {
    environmentReservationPersistence.findNearestComing(environmentId, applicationId, date)
  }

}
