package com.xebialabs.xlrelease.environments.service

import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.ExtensionCommand
import com.xebialabs.xlrelease.actors.extension.ActorExtensionHandlerFactory
import com.xebialabs.xlrelease.actors.{ActorSystemHolder, ReleaseActorService, ReleaseExecutionActor}
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.environments.events.{EnvironmentReservationCreatedEvent, EnvironmentReservationDeletedEvent, EnvironmentReservationUpdatedEvent}
import com.xebialabs.xlrelease.environments.service.EnvironmentReservationEventHandler.StartTaskNowIfNecessary
import com.xebialabs.xlrelease.events.{EventBus, Subscribe}
import com.xebialabs.xlrelease.repository.{Ids, TaskRepository}
import com.xebialabs.xlrelease.service.ExecutionService
import com.xebialabs.xlrelease.user.User
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.apache.pekko.actor.Actor.Receive
import org.apache.pekko.actor.{Actor, ActorRef}
import org.springframework.stereotype.Service

import jakarta.annotation.{PostConstruct, PreDestroy}
import scala.jdk.CollectionConverters._

object EnvironmentReservationEventHandler {

  case class StartTaskNowIfNecessary(taskId: String) extends ExtensionCommand

}

@Service
class EnvironmentReservationEventHandler(val actorSystemHolder: ActorSystemHolder,
                                         val eventBus: EventBus,
                                         val releaseActorService: ReleaseActorService,
                                         val taskRepository: TaskRepository,
                                         val executionService: ExecutionService) extends ActorExtensionHandlerFactory with Logging {

  @PostConstruct
  def register(): Unit = {
    eventBus.register(this)
    logger.debug("EnvironmentReservation event handler registered in event bus.")
  }

  @PreDestroy
  def unregister(): Unit = {
    eventBus.deregister(this)
    logger.debug("EnvironmentReservation event handler de-registered from event bus.")
  }

  @Subscribe
  @Timed
  def onEnvironmentReservationCreated(event: EnvironmentReservationCreatedEvent): Unit = {
    taskRepository.findPendingTasksWithFacets().asScala.foreach((taskId: String) => {
      releaseActorService.executeCommandAsync(Ids.releaseIdFrom(taskId), StartTaskNowIfNecessary(taskId))
    })
  }

  @Subscribe
  @Timed
  def onEnvironmentReservationUpdated(event: EnvironmentReservationUpdatedEvent): Unit = {
    taskRepository.findPendingTasksWithFacets().asScala.foreach((taskId: String) => {
      releaseActorService.executeCommandAsync(Ids.releaseIdFrom(taskId), StartTaskNowIfNecessary(taskId))
    })
  }

  @Subscribe
  @Timed
  def onEnvironmentReservationDeleted(event: EnvironmentReservationDeletedEvent): Unit = {
    taskRepository.findPendingTasksWithFacets().asScala.foreach((taskId: String) => {
      releaseActorService.executeCommandAsync(Ids.releaseIdFrom(taskId), StartTaskNowIfNecessary(taskId))
    })
  }


  override def getHandler(self: ActorRef, sender: () => ActorRef, release: => Release): Receive = {
    case StartTaskNowIfNecessary(taskId) => {
      release.getAllTasks.asScala.find((task) => {
        Ids.getFolderlessId(task.getId).equals(taskId)
      }).foreach((task) => {
        if (task.isPostponedUntilEnvironmentsAreReserved && task.getStatus == TaskStatus.PENDING) {
          logger.debug(s"Starting pending task ${taskId} because of event reservation change");
          executionService.startPendingTask(task.getId(), release, null, User.SYSTEM)
        }
      })
    }
  }

  override def supports(clazz: Class[_ <: Actor]): Boolean = classOf[ReleaseExecutionActor].isAssignableFrom(clazz)
}
