import React from 'react';
import { EnvironmentFormComponent } from './environment-form-component';
import { DotInputSelect, DotInputText } from '@digital-ai/dot-components';
import { environmentForm } from '../ducks/environment-form.reducer';
import { ReactWrapper, mountWithStoreAndTheme } from '@xlr-ui/tests/unit/testing-utils';
import { environmentState, envStateWithModel, model, stageList } from '../__mocks__/environment-list.mocks';
import { RootState } from '@xlr-ui/app/js/store.types';
import WidgetLinkedLabels from '../../shared/widgets/widget-linked-labels';
import { SidebarLabelList } from './sidebar-label-list';
import { labelsState } from '../ducks/labels.reducer';

const { searchLabels, saveLabel } = labelsState.actions;
const { setEnvironmentModel, setSaveEnabled } = environmentForm.actions;

describe('Environment edit form component', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();
    const mount = (state: RootState = envStateWithModel) => {
        wrapper = mountWithStoreAndTheme(<EnvironmentFormComponent environmentId={'envid'} folderId={''} />, dispatch, state);
    };

    beforeEach(() => {
        mount();
    });

    it('should render the form with model', () => {
        const form = wrapper.find('.environment-form');
        expect(form).toExist();

        const formFields = form.find(DotInputText);
        expect(formFields.exists()).toBeTruthy();
        expect(formFields).toHaveLength(2);
        const formName = formFields.at(0);
        expect(formName.prop('name')).toBe('environment-form-title');

        const formDescription = formFields.at(1);
        expect(formDescription.prop('name')).toBe('env-desc-input');

        const select = wrapper.find(DotInputSelect);
        expect(select.prop('name')).toBe('envStageSelector');
    });

    it('should use the proper callbacks', () => {
        const form = wrapper.find('.environment-form');
        const formFields = form.find(DotInputText);
        const formName = formFields.at(0);

        formName.invoke('onChange')({ target: { value: 'new name' } });
        expect(dispatch).toBeCalledWith(setEnvironmentModel({ ...model, title: 'new name' }));

        const formDescription = formFields.at(1);
        formDescription.invoke('onChange')({ target: { value: 'updated description' } });
        expect(dispatch).toBeCalledWith(setEnvironmentModel({ ...model, description: 'updated description' }));
    });

    it('should be not valid if name already exists', () => {
        const title = wrapper.find('h4.page-title');
        expect(title).toHaveLength(2);
        expect(title.at(0).text()).toBe('Environments / my name');
        const form = wrapper.find('.environment-form');
        const formFields = form.find(DotInputText);
        const formName = formFields.at(0);

        formName.invoke('onChange')({ target: { value: 'new name' } });
        expect(dispatch).toHaveBeenCalledWith(
            setEnvironmentModel({
                id: 'envid',
                folderId: '',
                title: 'new name',
                description: 'my description',
                stage: { id: 'EnvironmentStage1', title: 'test' },
                labels: [],
            }),
        );
        expect(dispatch).toHaveBeenCalledWith(setSaveEnabled(true));
        formName.invoke('onChange')({ target: { value: 'used' } });
        expect(dispatch).toHaveBeenCalledWith(
            setEnvironmentModel({
                id: 'envid',
                folderId: '',
                title: 'used',
                description: 'my description',
                stage: { id: 'EnvironmentStage1', title: 'test' },
                labels: [],
            }),
        );
        expect(dispatch).toHaveBeenCalledWith(setSaveEnabled(false));
    });

    it('should be not valid if stage is empty', () => {
        mount(environmentState);

        expect(dispatch).toHaveBeenCalledWith(setSaveEnabled(false));
    });

    it('should call pristine changed', () => {
        mount();
        wrapper.find(WidgetLinkedLabels).invoke('onDelete')({ id: '', color: '', title: '' });
    });

    it('should leave same stage', () => {
        mount();
        const select = wrapper.find(DotInputSelect);
        select.invoke('onChange')({ target: { value: 'changedStage' } });
        expect(dispatch).toHaveBeenCalledWith(setEnvironmentModel(model));
    });

    it('should change stage', () => {
        mount();
        const select = wrapper.find(DotInputSelect);
        select.invoke('onChange')({ target: { value: 'prod' } });
        expect(dispatch).toHaveBeenCalledWith(setEnvironmentModel({ ...model, stage: stageList[1] }));
    });

    it('should save label', () => {
        mount();
        const sidebar = wrapper.find(SidebarLabelList);
        const newLabel = { id: '', title: 'lbl', color: 'blue' };
        sidebar.invoke('saveLabel')(newLabel);
        expect(dispatch).toHaveBeenCalledWith(saveLabel(newLabel));
        expect(dispatch).toHaveBeenCalledWith(searchLabels(''));
    });
});
