package com.xebialabs.xlrelease.environments.repository

import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.domain.environments.{Application, Environment, EnvironmentLabel, EnvironmentStage}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.{ApplicationPersistence, EnvironmentPersistence, toDisplayId}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.data.RowWithContent
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper

import scala.collection.mutable
import scala.jdk.CollectionConverters._

package object sql {
  //noinspection ScalaStyle
  def mapApplicationContent(row: RowWithContent)
                           (implicit environmentPersistence: EnvironmentPersistence,
                            environmentStageRepository: EnvironmentStageRepository,
                            environmentLabelRepository: EnvironmentLabelRepository,
                            repositoryAdapter: SqlRepositoryAdapter,
                            cachedApps: mutable.Map[CiId, Application] = mutable.Map.empty[CiId, Application],
                            cachedEnvs: mutable.Map[CiId, Environment] = mutable.Map.empty[CiId, Environment],
                            cachedStages: mutable.Map[CiId, EnvironmentStage] = mutable.Map.empty[CiId, EnvironmentStage],
                            cachedLabels: mutable.Map[CiId, EnvironmentLabel] = mutable.Map.empty[CiId, EnvironmentLabel]
                           ): Application =
    cachedApps.getOrElseUpdate(row.id, {
      val ci = mapContent[Application](row)
      ci.setId(toDisplayId(ci.getId))
      ci.setEnvironments(ci.getEnvironments.asScala.map(e => mapEnvironment(e.getId)).asJava)
      ci
    })

  def mapEnvironmentContent(row: RowWithContent)
                           (implicit environmentStageRepository: EnvironmentStageRepository,
                            environmentLabelRepository: EnvironmentLabelRepository,
                            repositoryAdapter: SqlRepositoryAdapter,
                            cachedEnvs: mutable.Map[CiId, Environment] = mutable.Map.empty[CiId, Environment],
                            cachedStages: mutable.Map[CiId, EnvironmentStage] = mutable.Map.empty[CiId, EnvironmentStage],
                            cachedLabels: mutable.Map[CiId, EnvironmentLabel] = mutable.Map.empty[CiId, EnvironmentLabel]
                           ): Environment =
    cachedEnvs.getOrElseUpdate(row.id, {
      val ci = mapContent[Environment](row)
      ci.setId(toDisplayId(ci.getId))
      ci.setStage(mapStage(ci.getStage.getId))
      ci.setLabels(ci.getLabels.asScala.map(l => mapLabels(l.getId)).asJava)
      ci
    })


  //noinspection ScalaStyle
  def mapApplication(id: CiId)
                    (implicit applicationPersistence: ApplicationPersistence,
                     environmentPersistence: EnvironmentPersistence,
                     environmentStageRepository: EnvironmentStageRepository,
                     environmentLabelRepository: EnvironmentLabelRepository,
                     repositoryAdapter: SqlRepositoryAdapter,
                     cachedApps: mutable.Map[CiId, Application] = mutable.Map.empty[CiId, Application],
                     cachedEnvs: mutable.Map[CiId, Environment] = mutable.Map.empty[CiId, Environment],
                     cachedStages: mutable.Map[CiId, EnvironmentStage] = mutable.Map.empty[CiId, EnvironmentStage],
                     cachedLabels: mutable.Map[CiId, EnvironmentLabel] = mutable.Map.empty[CiId, EnvironmentLabel]): Application =
    cachedApps.getOrElseUpdate(id,
      mapApplicationContent(applicationPersistence.findById(id)
        .getOrElse(throw new IllegalStateException(s"Cannot find environment $id"))))

  def mapEnvironment(id: CiId)
                    (implicit environmentPersistence: EnvironmentPersistence,
                     environmentStageRepository: EnvironmentStageRepository,
                     environmentLabelRepository: EnvironmentLabelRepository,
                     repositoryAdapter: SqlRepositoryAdapter,
                     cachedEnvs: mutable.Map[CiId, Environment] = mutable.Map.empty[CiId, Environment],
                     cachedStages: mutable.Map[CiId, EnvironmentStage] = mutable.Map.empty[CiId, EnvironmentStage],
                     cachedLabels: mutable.Map[CiId, EnvironmentLabel] = mutable.Map.empty[CiId, EnvironmentLabel]): Environment =
    cachedEnvs.getOrElseUpdate(id,
      mapEnvironmentContent(environmentPersistence.findById(id)
        .getOrElse(throw new IllegalStateException(s"Cannot find environment $id"))))

  def mapStage(stageId: CiId)(implicit environmentStageRepository: EnvironmentStageRepository,
                              cachedStages: mutable.Map[CiId, EnvironmentStage] = mutable.Map.empty[CiId, EnvironmentStage]): EnvironmentStage =
    cachedStages.getOrElseUpdate(stageId, environmentStageRepository.findById(stageId))

  def mapLabels(labelId: CiId)(implicit environmentLabelRepository: EnvironmentLabelRepository,
                               cachedLabels: mutable.Map[CiId, EnvironmentLabel] = mutable.Map.empty[CiId, EnvironmentLabel]): EnvironmentLabel =
    cachedLabels.getOrElseUpdate(labelId, environmentLabelRepository.findById(labelId))

  private def mapContent[T <: BaseConfiguration](row: RowWithContent)(implicit repositoryAdapter: SqlRepositoryAdapter): T = {
    val ci = CiSerializerHelper.deserialize(row.json, repositoryAdapter).asInstanceOf[T]
    ci.setFolderId(row.folderId)
    ci
  }
}
