import React from 'react';
import { ApplicationEditHeaderComponent } from './application-edit-header-component';
import { SidebarEnvironmentList } from './form/sidebar-environment-list';
import { ApplicationFormComponent } from './form/application-form-component';
import { mountWithStoreAndTheme, preparePortalContainer, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { ApplicationEditPageComponent } from './application-edit-page-component';
import { RootState } from '@xlr-ui/app/js/store.types';
import {
    allApplications,
    appInFolder,
    appsInFolder,
    environmentsInFolder,
    formApplications,
    formEnvironments,
    formStages,
} from '../__mocks__/application-list.mocks';
import { Provider } from 'react-redux';
import { applicationForm } from '../ducks/application-form.reducer';

const { initForm, onSave, onUpdate, setApplicationFolder } = applicationForm.actions;

describe('Application edit page component', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();

    const beginState: RootState = {
        applicationList: {
            applications: allApplications,
            deleteModalIsOpen: false,
        },
        folderId: '',
        applicationForm: {
            stages: formStages,
            environments: formEnvironments,
            application: formApplications[0],
            applications: formApplications,
        },
    };

    const stateWithFolder: RootState = {
        applicationList: {
            applications: allApplications,
            deleteModalIsOpen: false,
        },
        folderId: '',
        applicationForm: {
            stages: formStages,
            environments: environmentsInFolder,
            application: appInFolder,
            applications: appsInFolder,
        },
    };

    const mount = (appState: RootState = beginState, appId = 'Application1', folderId = '') => {
        wrapper = mountWithStoreAndTheme(<ApplicationEditPageComponent {...{ applicationId: appId, folderId }} />, dispatch, appState);
    };

    beforeEach(() => {
        preparePortalContainer();
        mount();
    });

    afterEach(() => {
        wrapper.unmount();
    });

    it('should render properly', () => {
        expect(wrapper.find(Provider)).toExist();
        expect(wrapper.find(ApplicationEditPageComponent)).toExist();
        expect(dispatch).toBeCalledWith(initForm({ appId: 'Application1', folderId: '' }));
    });

    it('should initialize state on folder', () => {
        const appId = 'Application1';
        const folderId = 'Folder42';
        mount(beginState, appId, folderId);
        expect(dispatch).toBeCalledWith(initForm({ appId, folderId }));
    });

    it('should create all subcomponents', () => {
        const header = wrapper.find(ApplicationEditHeaderComponent);
        expect(header.prop('saveCaption')).toBe('Save');

        const environmentSidebar = wrapper.find(SidebarEnvironmentList);
        expect(environmentSidebar).toExist();
        expect(environmentSidebar.prop('stages')).toMatchObject(beginState.applicationForm.stages);
        expect(environmentSidebar.prop('environments')).toMatchObject([beginState.applicationForm.environments[1]]);

        const sidebarLayout = wrapper.find('.application-form-sidebar');
        expect(sidebarLayout).toExist();

        const form = wrapper.find(ApplicationFormComponent);
        expect(form).toExist();
    });

    it('should handle the callbacks', () => {
        const header = wrapper.find(ApplicationEditHeaderComponent);
        header.invoke('onSave')();
        expect(dispatch).toBeCalledWith(setApplicationFolder(''));
        expect(dispatch).toBeCalledWith(onSave(formApplications[0]));
        const form = wrapper.find(ApplicationFormComponent);
        form.invoke('onFormChange')(formApplications[0]);
        expect(dispatch).toBeCalledWith(onUpdate({ app: formApplications[0], folderId: '' }));
        const sidebar = wrapper.find(SidebarEnvironmentList);
        sidebar.invoke('onClickEnvironment')(formEnvironments[1]);
        expect(dispatch).toHaveBeenCalledWith(
            onUpdate({
                app: {
                    id: 'Application1',
                    title: 'App1',
                    folderId: '',
                    environments: formEnvironments,
                },
                folderId: '',
            }),
        );
    });

    it('should handle the callbacks for folder application', () => {
        const folderId = 'Folder42';
        const application = { id: 'Application1', title: 'Application1', folderId, environments: [] };
        mount(stateWithFolder, application.id, folderId);

        const header = wrapper.find(ApplicationEditHeaderComponent);
        header.invoke('onSave')();
        expect(dispatch).toBeCalledWith(setApplicationFolder(folderId));
        expect(dispatch).toBeCalledWith(onSave(application));
        const form = wrapper.find(ApplicationFormComponent);
        form.invoke('onFormChange')(application);
        expect(dispatch).toBeCalledWith(onUpdate({ app: application, folderId }));
    });
});
