package com.xebialabs.xlrelease.dsl.service.renderer

import com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.domain.ReleaseTrigger
import com.xebialabs.xlrelease.domain.variables._
import com.xebialabs.xlrelease.variable.VariableHelper.withoutVariableSyntax
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._
import scala.reflect.runtime.universe._

class VariableRenderer[T <: ConfigurationItem](dslKeyword: String)(implicit clazzTag: TypeTag[T]) extends BaseTypedCiRenderer[T](dslKeyword)(clazzTag) {
  override def ignoredProperties: List[String] = List("key") ++ super.ignoredProperties

  override def renderTitle(ci: ConfigurationItem): String = if (ci.hasProperty("key")) s"(${string(ci.getProperty("key"))})" else ""

  override def nameMapper = Map("requiresValue" -> "required")

  override protected def shouldRender(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Boolean = {
    val isReleaseTrigger = rendererContext.parent.exists(_.getType.isSubTypeOf(Type.valueOf(classOf[ReleaseTrigger])))
    if (rendererContext.shouldRenderAsTemplate && isReleaseTrigger) {
      pd.getName == "value" && super.shouldRender(rendererContext, ci, pd)
    } else {
      super.shouldRender(rendererContext, ci, pd)
    }
  }
}

@Component
class BooleanVariableRenderer extends VariableRenderer[BooleanVariable]("booleanVariable")

@Component
class IntegerVariableRenderer extends VariableRenderer[IntegerVariable]("integerVariable")

@Component
class PasswordVariableRenderer extends VariableRenderer[PasswordStringVariable]("passwordVariable") {
  override def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    pd.getKind match {
      case CI if pd.getName == "externalVariableValue" => Some(getDefaultRenderer.dsl(rendererContext, pd.get(ci).asInstanceOf[ConfigurationItem]))
      case _ => super.renderProperty(rendererContext, ci, pd)
    }
  }
}

@Component
class ExternalVariableValueRenderer extends BaseTypedCiRenderer[ExternalVariableValue]("externalVariableValue") {
  override def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {

    pd.getKind match {
      case CI if pd.getName == "server" =>
        val value = pd.get(ci)
        if (value == null) {
          None
        } else {

          val max = rendererContext.ciReferences.asScala.keys.map {
            case x if x.startsWith(pd.getName) => Integer.valueOf(x.replace(pd.getName, ""))
            case _ => Integer.valueOf(0)
          }.toSeq match {
            case Nil => Integer.valueOf(0)
            case y => y.max
          }
          val variableId = s"${pd.getName}${max + 1}"
          rendererContext.ciReferences.put(variableId, value.asInstanceOf[ConfigurationItem])
          Some(s"${formatName(pd)} $variableId")
        }
      case _ => super.renderProperty(rendererContext, ci, pd)
    }
  }
}

@Component
class StringVariableRenderer extends VariableRenderer[StringVariable]("stringVariable") {

  override def dsl(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    val bci = ci.asInstanceOf[StringVariable]
    val keyword = if (bci.getValueProvider == null) "stringVariable" else "listBoxVariable"
    s"""$keyword('${bci.getKey}') {
       |${renderProperties(rendererContext, ci)}
       |}""".stripMargin
  }

  override def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    pd.getKind match {
      case CI if pd.getName == "valueProvider" => Some(getDefaultRenderer.dsl(rendererContext, pd.get(ci).asInstanceOf[ConfigurationItem]))
      case _ => super.renderProperty(rendererContext, ci, pd)
    }
  }
}

@Component
class ListStringVariableRenderer extends VariableRenderer[ListStringVariable]("listVariable") {
  override def dsl(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    val bci = ci.asInstanceOf[ListStringVariable]
    val keyword = if (bci.getValueProvider == null) "listVariable" else "multiSelectListBoxVariable"
    s"""$keyword('${bci.getKey}') {
       |${renderProperties(rendererContext, ci)}
       |}""".stripMargin
  }

  override def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    pd.getKind match {
      case CI if pd.getName == "valueProvider" => Some(getDefaultRenderer.dsl(rendererContext, pd.get(ci).asInstanceOf[ConfigurationItem]))
      case _ => super.renderProperty(rendererContext, ci, pd)
    }
  }
}

@Component
class SetStringVariableRenderer extends VariableRenderer[SetStringVariable]("setVariable")

@Component
class MapVariableRenderer extends VariableRenderer[MapStringStringVariable]("mapVariable")

@Component
class DateVariableRenderer extends VariableRenderer[DateVariable]("dateVariable")

@Component
class ValueProviderRenderer extends BaseTypedCiRenderer[ValueProviderConfiguration](null) {
  override def dsl(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    if (ci.getType.instanceOf(Type.valueOf(classOf[ListOfStringValueProviderConfiguration]))) {
      val bci = ci.asInstanceOf[ListOfStringValueProviderConfiguration]
      if (bci.getVariableKey == null) {
        s"possibleValues ${bci.getValues.asScala.map(v => s"${string(v)}").mkString(", ")}"
      } else {
        s"possibleValues variable('${withoutVariableSyntax(bci.getVariableKey)}')"
      }
    } else {
      // we don't show possible values in scripted value providers as it doesn't make sense to show it in the DSL
      s"valueProvider '${ci.getType}'"
    }
  }
}
