package com.xebialabs.xlrelease.dsl.service.renderer

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind._
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.variable.VariableHelper.withoutVariableSyntax

import java.text.SimpleDateFormat
import java.util
import java.util.Date
import scala.jdk.CollectionConverters._

private[renderer] trait BaseCiRenderer {

  def ignoredProperties: List[String] = List()

  def propertyOrder: Map[String, Int] = Map()

  def defaultPropertyOrder: Int = 10

  def nameMapper: Map[String, String] = Map()

  def renderCi(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    s"""|${ci.getType.getName} {
        |${renderProperties(rendererContext, ci)}
        |}""".stripMargin
  }

  def renderProperties(rendererContext: DslRendererContext, ci: ConfigurationItem): String = {
    rendererContext.ancestors = ci :: rendererContext.ancestors
    val props = for {
      pd <- sortedProperties(ci)
      if shouldRender(rendererContext, ci, pd)
    } yield renderPropertyTemplate(rendererContext, ci, pd)
    rendererContext.ancestors = rendererContext.ancestors.tail
    props.flatten.mkString("\n")
  }

  //noinspection ScalaStyle
  private def renderPropertyTemplate(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    val propertyName = pd.getName
    val propertyNamePrefix = s"$propertyName."
    val originalVariableMapping = rendererContext.currentCiVariableMapping

    if (originalVariableMapping.containsKey(propertyName)) {
      Some(s"${formatName(pd)} variable('${withoutVariableSyntax(originalVariableMapping.get(propertyName))}')")
    } else ci match {
      case variable: Variable if originalVariableMapping.containsKey(variable.getKey) =>
        Some(s"variable('${withoutVariableSyntax(originalVariableMapping.get(variable.getKey))}')")
      case _ =>
        if (pd.getKind == CI) {
          // for now we don't have to handle collections of CIs
          val childCiVariableMapping = originalVariableMapping.asScala.view.filterKeys(_.startsWith(propertyNamePrefix)).map {
            case (k, v) => k.stripPrefix(propertyNamePrefix) -> v
          }.toMap
          rendererContext.currentCiVariableMapping = childCiVariableMapping.asJava
        }

        val renderedDsl = renderProperty(rendererContext, ci, pd)
        rendererContext.currentCiVariableMapping = originalVariableMapping
        renderedDsl
    }
  }

  // scalastyle:off cyclomatic.complexity
  def renderProperty(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    val value = pd.get(ci)
    if (value == null) {
      None
    } else {
      pd.getKind match {
        case DATE => Some(s"""${formatName(pd)} Date.parse("yyyy-MM-dd'T'HH:mm:ssZ", ${string(format(value.asInstanceOf[Date]))})""")
        case BOOLEAN => Some(s"${formatName(pd)} $value")
        case ENUM if value.isInstanceOf[String] => Some(s"${formatName(pd)} ${string(value.toString)}")
        case ENUM => Some(s"${formatName(pd)} ${value.getClass.getName}.$value")
        case INTEGER =>
          if (value.asInstanceOf[Integer] < 0) {
            Some(s"${formatName(pd)} ($value)")
          } else {
            Some(s"${formatName(pd)} $value")
          }
        case STRING if pd.isPassword => Some(s"${formatName(pd)} ${string(PasswordEncrypter.getInstance.ensureEncrypted(value.toString))}")
        case STRING => Some(s"${formatName(pd)} ${string(value.toString)}")
        case LIST_OF_CI | SET_OF_CI =>
          val items = value.asInstanceOf[util.Collection[ConfigurationItem]]
          renderCiCollection(rendererContext, items, pd)
        case SET_OF_STRING => renderStringCollection(ci, pd)
        case LIST_OF_STRING => renderStringCollection(ci, pd)
        case MAP_STRING_STRING => renderMapStringString(ci, pd)
        case CI =>
          val referencedCi = value.asInstanceOf[ConfigurationItem]
          if (referencedCi.hasProperty("title")) {
            val title = referencedCi.getProperty[String]("title")
            Some(s"${formatName(pd)} '$title'")
          } else {
            if (pd.isNested) {
              Some(renderCi(rendererContext, referencedCi))
            } else {
              None
            }
          }
        case _ => Some(s"// no DSL renderer found for property '${pd.getName}' of type '${ci.getType}'")
      }
    }
  }

  protected def renderCiCollection(rendererContext: DslRendererContext, items: util.Collection[_ <: ConfigurationItem], pd: PropertyDescriptor): Option[String] = {
    if (items.isEmpty) {
      None
    } else {
      Some(
        s"""${formatName(pd)} {
           |${items.asScala.map(renderCi(rendererContext, _)).mkString("\n")}
           |}""".stripMargin
      )
    }
  }

  private def renderMapStringString(ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    val mapStringString = pd.get(ci).asInstanceOf[util.Map[String, String]]
    if (mapStringString.isEmpty) None else Some(s"${formatName(pd)} ${mapStringString.asScala.map { case (k, v) => s"${string(k)}:${string(v)}" }.mkString(",")}")
  }

  private def renderStringCollection(ci: ConfigurationItem, pd: PropertyDescriptor): Option[String] = {
    val stringCollection = pd.get(ci).asInstanceOf[util.Collection[String]]
    if (!stringCollection.isEmpty) {
      Some(s"${formatName(pd)} ${stringCollection.asScala.map(v => s"${string(v)}").mkString(", ")}")
    } else {
      None
    }
  }

  private def sortedProperties(ci: ConfigurationItem): List[PropertyDescriptor] =
    ci.getType.getDescriptor.getPropertyDescriptors.asScala.toList.sortWith(sortByPropertyOrderOf)


  private def sortByPropertyOrderOf = (pd1: PropertyDescriptor, pd2: PropertyDescriptor) => {
    propertyOrder.getOrElse(pd1.getName, defaultPropertyOrder) < propertyOrder.getOrElse(pd2.getName, defaultPropertyOrder)
  }

  protected def shouldRender(rendererContext: DslRendererContext, ci: ConfigurationItem, pd: PropertyDescriptor): Boolean = {
    val isInVariableMapping = rendererContext.currentCiVariableMapping.containsKey(pd.getName) ||
      (pd.getName == "value" && ci.isInstanceOf[Variable] && rendererContext.currentCiVariableMapping.containsKey(ci.asInstanceOf[Variable].getKey))
    val isNotIgnored = !ignoredProperties.contains(pd.getName)
    val isNotInternal = "internal" != pd.getCategory
    val isNotHidden = !pd.isHidden
    val isNotNull = pd.get(ci) != null
    val hasNonDefaultValue = pd.get(ci) != pd.getDefaultValue
    isInVariableMapping || (isNotInternal && isNotHidden && isNotNull && hasNonDefaultValue && isNotIgnored)
  }

  private def format(date: Date): String = {
    val sdf: SimpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ")
    sdf.format(date)
  }

  private val groovyKeywords = Array("as", "assert", "break", "case", "catch", "class", "const", "continue", "def",
    "default", "do", "else", "enum", "extends", "false", "finally", "for", "goto", "if", "implements", "import", "in",
    "instanceof", "interface", "new", "null", "package", "return", "super", "switch", "this", "throw", "throws",
    "trait", "true", "try", "while")

  protected def formatName(pd: PropertyDescriptor): String = {
    val name = if (pd.getName.charAt(0).isUpper || groovyKeywords.contains(pd.getName)) s"'${pd.getName}'" else pd.getName
    nameMapper.getOrElse(pd.getName, name)
  }
}
