package com.xebialabs.xlrelease.dsl.service;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.api.internal.InternalMetadata;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.dsl.DependencyTargetSpec;
import com.xebialabs.xlrelease.dsl.IdDependencyTargetSpec;
import com.xebialabs.xlrelease.dsl.TargetSpec;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.repository.Page;
import com.xebialabs.xlrelease.repository.PlanItemRepository;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.FolderService;
import com.xebialabs.xlrelease.service.ReleaseService;

import static com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_RELEASE;
import static org.springframework.util.StringUtils.hasText;

@Component
public class DependencyCiProcessor implements CiProcessor<Dependency> {
    private ReleaseService releaseService;
    private PlanItemRepository planItemRepository;
    private FolderService folderService;
    private PermissionChecker permissionChecker;

    @Autowired
    public DependencyCiProcessor(final ReleaseService releaseService,
                                 final PlanItemRepository planItemRepository,
                                 final FolderService folderService,
                                 final PermissionChecker permissionChecker) {
        this.releaseService = releaseService;
        this.planItemRepository = planItemRepository;
        this.folderService = folderService;
        this.permissionChecker = permissionChecker;
    }


    @Override
    public void process(final DslProcessingContext processingContext, final Dependency ci) {
        Map<String, InternalMetadata> $metadata = ci.get$metadata();

        if (null != $metadata) {
            TargetSpec targetSpec = (TargetSpec) $metadata.get("targetSpec");
            if (null != ci.getTargetId() && null != targetSpec) {
                throw new DslError("Set either a variable or target for a dependency in gate task.");
            }
            if (null != targetSpec) {
                if (targetSpec instanceof DependencyTargetSpec) {
                    String containerId = processingContext.getContainer() != null ? processingContext.getContainer().getId() : ROOT_FOLDER_ID;
                    String defaultFolderId = Ids.findFolderId(containerId);
                    ci.setTarget(resolveTargetFromPath(defaultFolderId, (DependencyTargetSpec) targetSpec));
                } else if (targetSpec instanceof IdDependencyTargetSpec) {
                    ci.setTarget(resolveTargetFromId((IdDependencyTargetSpec) targetSpec));
                }
            }
        }
    }

    @Override
    public Type getType() {
        return Type.valueOf(Dependency.class);
    }

    private PlanItem resolveTargetFromId(IdDependencyTargetSpec targetSpec) {
        String targetPlanItemId = targetSpec.getTargetId();
        try {
            return planItemRepository.findById(targetPlanItemId);
        } catch (NotFoundException ignore) {
            throw new DslError("No target found with ID: '%s'", targetPlanItemId);
        }
    }

    private PlanItem resolveTargetFromPath(final String defaultFolderId, final DependencyTargetSpec targetSpec) {
        String releaseTitleOrId = targetSpec.getRelease();
        if (hasText(releaseTitleOrId)) {
            Release release = resolveRelease(defaultFolderId, releaseTitleOrId);
            String phaseTitleOrId = targetSpec.getPhase();
            if (hasText(phaseTitleOrId)) {
                Phase phase = getPhaseFromRelease(release, phaseTitleOrId);
                String taskTitleOrId = targetSpec.getTask();
                if (hasText(taskTitleOrId)) {
                    return getTaskFromPhase(phase, taskTitleOrId);
                } else {
                    return phase;
                }
            } else {
                return release;
            }
        }
        throw new DslError("No target found with path '%s'.", getPathFromTarget(targetSpec));
    }

    private Release resolveRelease(final String defaultFolderId, final String releaseTitleOrId) {
        Release release;
        try {
            release = releaseService.findById(releaseTitleOrId);
        } catch (NotFoundException ignore) {
            release = getReleaseFromPath(defaultFolderId, releaseTitleOrId);
        }
        return release;
    }

    private Task getTaskFromPhase(final Phase phase, final String taskTitleOrId) {
        return phase.getTasks()
                .stream()
                .filter(task -> taskTitleOrId.equals(task.getTitle()) || taskTitleOrId.equals(task.getId()))
                .findAny()
                .orElseThrow(() -> new DslError("No task found with title/id '%s'.", taskTitleOrId));
    }

    private Phase getPhaseFromRelease(final Release release, final String phaseTitleOrId) {
        return release.getPhases()
                .stream()
                .filter(phase -> phaseTitleOrId.equals(phase.getTitle()) || phaseTitleOrId.equals(phase.getId()))
                .findAny()
                .orElseThrow(() -> new DslError("No phase found with title/id '%s'.", phaseTitleOrId));
    }

    private Release getReleaseFromPath(final String defaultFolderId, final String releasePath) {
        Optional<String> folderPath = PathUtils.folderPath(releasePath);
        String folderId = folderPath.map(path -> folderService.findByPath(path, Page.DEFAULT_DEPTH()).getId()).orElse(defaultFolderId);
        String releaseTitle = releaseTitle(releasePath);
        List<Release> releases = this.releaseService.findReleasesByTitle(folderId, releaseTitle, 0, 2, Integer.MAX_VALUE);

        if (releases.isEmpty()) {
            throw new DslError("No release found with title '%s'.", releaseTitle);
        }
        if (releases.size() > 1) {
            throw new DslError("More than one release found with title '%s'.", releaseTitle);
        }
        releases = permissionChecker.filter(releases, VIEW_RELEASE);
        if (releases.isEmpty()) {
            throw new DslError("You don't have view permissions on the release with title '%s'.", releaseTitle);
        }
        return releases.get(0);
    }

    private String releaseTitle(final String fullReleasePath) {
        return PathUtils.releasePath(fullReleasePath).orElseThrow(() ->
                new DslError("Unable to find template title in '%s'", fullReleasePath));
    }

    private String getPathFromTarget(final DependencyTargetSpec targetSpec) {
        String path = "";
        if (null != targetSpec.getRelease()) {
            path += targetSpec.getRelease();
        }
        if (null != targetSpec.getPhase()) {
            path += "/" + targetSpec.getPhase();
        }
        if (null != targetSpec.getTask()) {
            path += "/" + targetSpec.getTask();
        }
        return path;
    }

}
