package com.xebialabs.xlrelease.export

import com.xebialabs.deployit.checks.Checks._
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.export.ReleasefileExporter.RELEASEFILE_FILENAME
import com.xebialabs.xlrelease.export.ReleasefileImporter.releaseFileImportScriptContext
import com.xebialabs.xlrelease.script.builder.ScriptContextBuilder
import com.xebialabs.xlrelease.script.groovy.GroovyScriptService
import com.xebialabs.xlrelease.script.{XlrScript, XlrScriptContext}
import org.springframework.stereotype.Component

import java.nio.charset.StandardCharsets
import java.util
import javax.script.ScriptContext
import scala.jdk.CollectionConverters._
import scala.util.Using

@Component
class ReleasefileImporter(groovyScriptService: GroovyScriptService) extends UnzippedTemplateImporter[ReleasefileTemplateContent]  {

  override def name(): String = {
    "Groovy DSL"
  }

  override def requiredFileName(): String = RELEASEFILE_FILENAME

  override def doReadTemplateContent(zipEntries: Map[String, LocalFile], importContext: TemplateImportContext): ReleasefileTemplateContent = {

    val dslContent: String = zipEntries.get(RELEASEFILE_FILENAME).map(OverthereUtils.read(_, StandardCharsets.UTF_8.name())).orNull

    checkArgument(dslContent != null, s"Missing file in template export : $RELEASEFILE_FILENAME. Check if file exists in the root of the archive.")

    ReleasefileTemplateContent(dslContent, zipEntries.asJava)
  }

  override def doUpgradeTemplateContent(templateContent: ReleasefileTemplateContent): util.List[String] = {
    // do not upgrade dsl files
    new util.ArrayList[String]()
  }

  override def doConvertToTemplate(templateContent: ReleasefileTemplateContent, destinationFolderId: String): Release = {
    // execute dslContent and create in-memory Release object
    Using.resource(releaseFileImportScriptContext(templateContent, destinationFolderId)) { scriptContext =>
      groovyScriptService.executeScript(scriptContext).asInstanceOf[Release]
    }
  }
}

case class ReleasefileTemplateContent(dslContent: String, localFiles: util.Map[String, LocalFile]) extends TemplateContent

object ReleasefileImporter {

  def releaseFileImportScriptContext(templateContent: ReleasefileTemplateContent, destinationFolderId: String): XlrScriptContext = {
    new ReleaseFileImportScriptContextBuilder(templateContent, destinationFolderId).build()
  }

  private class ReleaseFileImportScriptContextBuilder(templateContent: ReleasefileTemplateContent, destinationFolderId: String) extends ScriptContextBuilder {
    withScriptApi().withOpenApi().withGroovyImportDsl()

    override protected def doBuild(context: XlrScriptContext): Unit = {
      context.setAttribute("_importDestinationFolderId", destinationFolderId, ScriptContext.ENGINE_SCOPE)
      context.addScript(XlrScript.byContent(name = "<release_file_import>", content = templateContent.dslContent, wrap = false, checkPermissions = true))
    }

  }

}
