package com.xebialabs.xlrelease.lookup.delegate

import com.fasterxml.jackson.core.`type`.TypeReference
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, Delegate, Parameters}
import com.xebialabs.xlrelease.domain.Configuration
import com.xebialabs.xlrelease.lookup.api.internal.LookupResultElement
import com.xebialabs.xlrelease.runner.impl.{ContainerScriptException, RunnerScriptService}
import com.xebialabs.xlrelease.service.ConfigurationVariableService
import grizzled.slf4j.{Logger, Logging}

import java.util
import java.util.{List => JList, Map => JMap, Set => JSet}
import scala.jdk.CollectionConverters._

object RemoteScriptLookupDelegate extends BaseLookupDelegate with Logging {

  private val ScriptAttributeName = "script"
  private val ContainerConnectionAttributeName = "containerConnection"
  private val CapabilitiesPropertyName = "capabilities"
  // remove ugly '$' at the end of this singleton when logging
  override def logger: Logger = Logger(this.getClass.getName.dropRight(1))

  lazy val scriptService: RunnerScriptService = springBean[RunnerScriptService]
  private lazy val configurationVariableService = springBean[ConfigurationVariableService]

  @Delegate(name = "remoteScriptLookup")
  override def lookup(ci: ConfigurationItem, methodName: String, attrs: JMap[String, String], parameters: Parameters): JList[LookupResultElement] = {
    configurationVariableService.resolveFromCi(ci)()
    val script = if (attrs.containsKey(ScriptAttributeName)) {
      attrs.get(ScriptAttributeName)
    } else {
      throw ContainerScriptException("Container script missing in method delegate attribute setting")
    }
    val capabilities: JSet[String] = getCapabilities(ci, attrs)

    val inputParameters: JMap[String, AnyRef] = new util.HashMap[String, AnyRef]()
    inputParameters.put("_ci", ci) // writing it as json string
    inputParameters.put("_parameters", parameters) // writing it as json string
    inputParameters.put("_attributes", attrs) // writing it as json string

    scriptService.executeScript[JList[LookupResultElement]](
      script,
      capabilities.asScala.toSet,
      inputParameters,
      new TypeReference[JList[LookupResultElement]] {}
    )
  }

  private def getCapabilities(ci: ConfigurationItem, attrs: JMap[String, String]) = {
    if (ci.hasProperty(CapabilitiesPropertyName)) {
      ci.getProperty(CapabilitiesPropertyName).asInstanceOf[JSet[String]]
    } else {
      val connectionCapabilityProperty = if (attrs.containsKey(ContainerConnectionAttributeName)) {
        attrs.get(ContainerConnectionAttributeName)
      } else {
        throw ContainerScriptException("Container connection missing in method delegate attribute setting")
      }

      val capabilityConnection = if (ci.hasProperty(connectionCapabilityProperty)) {
        ci.getProperty(connectionCapabilityProperty).asInstanceOf[Configuration]
      } else {
        throw ContainerScriptException(s"Container connection CI missing, $ContainerConnectionAttributeName property " +
          s"points to $connectionCapabilityProperty which is missing in the CI")
      }

      val capabilities = if (capabilityConnection.hasProperty(CapabilitiesPropertyName)) {
        capabilityConnection.getProperty(CapabilitiesPropertyName).asInstanceOf[JSet[String]]
      } else {
        throw ContainerScriptException(s"Can't find capabilities for $connectionCapabilityProperty")
      }
      capabilities
    }
  }
}
