package com.xebialabs.xlrelease.pendo

import com.xebialabs.analytics.pendo.{PendoDataExtractor, PendoEventQueue}
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import grizzled.slf4j.Logging

@EventListener
class PendoRelevantEventsListener(pendoSwitch: PendoSwitch, eventQueue: PendoEventQueue, dataExtractors: Map[Class[_], PendoDataExtractor[_]]) extends Logging {

  @AsyncSubscribe
  def onReleaseCreated(event: ReleaseCreatedEvent): Unit = {
    if (!event.release.isTemplate) {
      processEventData(event)
    }
  }

  @AsyncSubscribe
  def onReleaseStarted(event: ReleaseStartedEvent): Unit = {
    processEventData(event)
  }

  @AsyncSubscribe
  def onReleaseFailed(event: ReleaseFailedEvent): Unit = {
    processEventData(event)
  }

  @AsyncSubscribe
  def onReleaseAborted(event: ReleaseAbortedEvent): Unit = {
    processEventData(event)
  }

  @AsyncSubscribe
  def onReleaseCompleted(event: ReleaseCompletedEvent): Unit = {
    processEventData(event)
  }

  private def processEventData(releaseEvent: ReleaseEvent): Unit = {
    if (pendoSwitch.isEnabled) {
      dataExtractors.get(releaseEvent.getClass) match {
        case Some(dataExtractor) =>
          dataExtractor.extractPendoData(releaseEvent) match {
            case Some(extractedData) => eventQueue.add(extractedData)
            case None => logger.warn(s"Unable to extract Pendo data for event ${releaseEvent.getClass.getName}, skipping")
          }
        case _ => logger.warn(s"Received unhandled Pendo event notification ${releaseEvent.getClass.getName}")
      }
    }
  }
}
