package com.xebialabs.xlrelease.pendo.metrics

import com.xebialabs.analytics.pendo.{PendoEvent, PendoEventQueue}
import com.xebialabs.xlplatform.cluster.ClusterMode
import com.xebialabs.xlrelease.pendo.PendoConstants._
import com.xebialabs.xlrelease.pendo.PendoSwitch
import com.xebialabs.xlrelease.pendo.utils.PendoUtils._
import com.xebialabs.xlrelease.support.report._
import com.xebialabs.xlrelease.utils.Limits.MAX_FOLDER_DEPTH
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired

import java.time.Instant
import scala.jdk.CollectionConverters.{ListHasAsScala, MapHasAsScala}

case class DataStatisticReports(releases: ReleaseUsageStatistics, tasks: TaskUsageStatistics, templates: TemplateUsageStatistics,
                                triggers: TriggerUsageStatistics, folders: FolderUsageStatistics, variables: VariableUsageStatistics,
                                users: UserUsageStatistics, dashboards: DashboardUsageStatistics, groups: ReleaseGroupUsageStatistics,
                                deliveries: DeliveryUsageStatistics, archive: ArchiveUsageStatistics, configuration: ConfigurationUsageStatistics,
                                applications: ApplicationUsageStatistics, environments: EnvironmentUsageStatistics,
                                calendarEntries: CalendarUsageStatistics, reports: ReportUsageStatistics)

class PendoMetricsService(eventQueue: PendoEventQueue, pendoSwitch: PendoSwitch) extends Logging {

  @Autowired
  var systemInformationServer: SystemInformationService = _

  def processMetrics(): Unit = {
    if (pendoSwitch.isEnabled) {
      try {
        val systemInfo = systemInformationServer.generateJsonReport(Seq("cluster", "general", "data", "installation", "license")).asScala.toMap
        val clusterStats = getStatisticsAs[ClusterUsageStatistics]("cluster", systemInfo)
        val generalStats = getStatisticsAs[GeneralUsageStatistics]("general", systemInfo)
        val licenseStats = getStatisticsAs[LicenseUsageStatistics]("license", systemInfo)
        val installationStats = getStatisticsAs[InstallationUsageStatistics]("installation", systemInfo)
        val usageStatsAsMap = getStatisticsAs[java.util.Map[String, Any]]("data", systemInfo).asScala.toMap
        val usageStats = extractUsageStatistics(usageStatsAsMap)

        eventQueue.add(generateReleaseMetricsEvent(usageStats.releases, usageStats.tasks, usageStats.templates, usageStats.triggers))
        eventQueue.add(generateTaskMetricsEvent(usageStats.tasks))
        eventQueue.add(generateUserMetricsEvent(usageStats.users))
        eventQueue.add(generateReportMetricsEvent(usageStats.reports))
        eventQueue.add(generateUsageMetricsEvent(usageStats.configuration, usageStats.folders, usageStats.variables, usageStats.dashboards,
          usageStats.groups, usageStats.deliveries, usageStats.applications, usageStats.environments, usageStats.calendarEntries))
        eventQueue.add(generateArchiveMetricsEvent(usageStats.archive))
        eventQueue.add(generateLicenseMetricsEvent(licenseStats))
        eventQueue.add(generateInstallationMetricsEvent(generalStats, clusterStats, installationStats))
      } catch {
        case t: Throwable => logger.warn(s"Unable to send metrics to Pendo, error: ${t.getMessage}")
      }
    }
  }

  def extractUsageStatistics(dataStatisticsReport: Map[String, Any]): DataStatisticReports = {
    DataStatisticReports(getStatisticsAs[ReleaseUsageStatistics]("releases", dataStatisticsReport),
      getStatisticsAs[TaskUsageStatistics]("tasks", dataStatisticsReport),
      getStatisticsAs[TemplateUsageStatistics]("templates", dataStatisticsReport),
      getStatisticsAs[TriggerUsageStatistics]("triggers", dataStatisticsReport),
      getStatisticsAs[FolderUsageStatistics]("folders", dataStatisticsReport),
      getStatisticsAs[VariableUsageStatistics]("variables", dataStatisticsReport),
      getStatisticsAs[UserUsageStatistics]("users", dataStatisticsReport),
      getStatisticsAs[DashboardUsageStatistics]("dashboards", dataStatisticsReport),
      getStatisticsAs[ReleaseGroupUsageStatistics]("groups", dataStatisticsReport),
      getStatisticsAs[DeliveryUsageStatistics]("deliveries", dataStatisticsReport),
      getStatisticsAs[ArchiveUsageStatistics]("archive", dataStatisticsReport),
      getStatisticsAs[ConfigurationUsageStatistics]("configurations", dataStatisticsReport),
      getStatisticsAs[ApplicationUsageStatistics]("applications", dataStatisticsReport),
      getStatisticsAs[EnvironmentUsageStatistics]("environments", dataStatisticsReport),
      getStatisticsAs[CalendarUsageStatistics]("calendarEntries", dataStatisticsReport),
      getStatisticsAs[ReportUsageStatistics]("report", dataStatisticsReport))
  }

  def getStatisticsAs[T](key: String, statMap: Map[String, Any]): T = Option(statMap.get(key)) match {
    case Some(stats) => stats.get.asInstanceOf[T]
    case None => throw new NoSuchElementException(s"Unable to find statistics data '${key}' in System Information report")
  }

  // TODO ted: try remove all these shuffling of data from one bucket into another

  def generateReleaseMetricsEvent(releases: ReleaseUsageStatistics, tasks: TaskUsageStatistics, templates: TemplateUsageStatistics,
                                  triggers: TriggerUsageStatistics): PendoEvent = {
    val properties = Map(
      TEMPLATES -> templates.templatesTotal,
      TEMPLATE_REVISIONS -> templates.templateRevisionsTotal,
      "triggers" -> triggers.triggersTotal,
      ACTIVE_TRIGGERS -> triggers.activeTriggersTotal,
      PLANNED_RELEASES -> releases.plannedReleasesTotal,
      IN_PROGRESS_RELEASES -> releases.inProgressReleasesTotal,
      FAILING_RELEASES -> releases.failingReleasesTotal,
      FAILED_RELEASES -> releases.failedReleasesTotal,
      PAUSED_RELEASES -> releases.pausedReleasesTotal,
      COMPLETED_RELEASES -> releases.completedReleasesTotal,
      ABORTED_RELEASES -> releases.abortedReleasesTotal,
      RELEASES_LAST_24_HOURS -> releases.releases24HoursTotal,
      RELEASES_LAST_30_DAYS -> releases.releases30DaysTotal,
      COMPLETION_RATE_MONTH_AGO -> releases.completionRateLastMonth,
      AVG_RELEASES_PER_DAY_LAST_YEAR -> releases.averageReleasesPerDayLastYear,
      AVG_RELEASES_PER_MONTH_LAST_YEAR -> releases.averageReleasesPerMonthLastYear,
      RELEASES_WITH_DEPENDENCIES -> releases.releasesWithDependencies,
      TOTAL_DEPENDENCIES -> releases.dependenciesTotal,
      AVG_DEPENDENCIES_PER_RELEASE -> releases.averageDependenciesPerReleases,
      MAX_DEPENDENCIES_PER_RELEASE -> releases.maxDependenciesInRelease,
      AVG_TASKS_PER_RELEASE -> tasks.averageTasksPerRelease,
      MAX_TASKS_IN_RELEASE -> tasks.maxTasksInRelease,
      RELEASE_TAGS -> releases.releaseTagsTotal
    ).toPendo

    PendoEvent(RELEASE_METRICS, properties, Instant.now.toEpochMilli)
  }

  def generateTaskMetricsEvent(tasks: TaskUsageStatistics): PendoEvent = {
    val properties = (Map(
      PLANNED_TASKS -> tasks.plannedTasksTotal,
      PENDING_TASKS -> tasks.pendingTasksTotal,
      IN_PROGRESS_TASKS -> tasks.inProgressTasksTotal,
      FAILED_TASKS -> tasks.failedTasksTotal,
      SKIPPED_TASKS -> tasks.skippedTasksTotal,
      COMPLETED_TASKS -> tasks.completedTasksTotal,
      ABORTED_TASKS -> tasks.abortedTasksTotal,
      "templateTasks" -> tasks.templateTasksTotal,
      AUTOMATION_PERCENTAGE_METRIC -> tasks.automationPercentage,
      TASK_TAGS -> tasks.taskTagsTotal,
      TASK_BACKUPS -> tasks.taskBackupsTotal,
      TASK_COMMENTS -> tasks.commentsTotal,
      TASK_ATTACHMENTS -> tasks.attachmentsTotal
    ) ++ createTop25TaskMetrics(tasks.taskTypeOccurrences.asScala.toList)).toPendo

    PendoEvent(TASK_METRICS, properties, Instant.now.toEpochMilli)
  }

  def generateUserMetricsEvent(users: UserUsageStatistics): PendoEvent = {
    val properties = Map(
      ALL_USERS -> users.usersTotal,
      ENABLED_USERS -> users.enabledUsersTotal,
      INTERNAL_USERS -> users.internalUsersTotal,
      EXTERNAL_USERS -> users.externalUsersTotal,
      ACTIVE_USERS_LAST_24_HOURS -> users.activeUsersLastDay,
      ACTIVE_USERS_LAST_30_DAYS -> users.activeUsersLast30Days,
      "activeUsersLast90Days" -> users.activeUsersLast90Days,
      GLOBAL_ROLES -> users.globalRolesTotal,
      TEAMS -> users.teamsTotal,
      ROLE_ROLES -> users.roleRolesTotal,
      ROLE_PRINCIPALS -> users.rolePrincipalsTotal,
      ROLE_PERMISSIONS -> users.rolePermissionsTotal
    ).toPendo

    PendoEvent(USER_METRICS, properties, Instant.now.toEpochMilli)
  }

  def generateReportMetricsEvent(reports: ReportUsageStatistics): PendoEvent = {
    val properties = Map(
      ACTIVITY_LOGS -> reports.activityLogsTotal,
      ATTRIBUTES_METRIC -> reports.attributesTotal,
      DEPLOYMENTS -> reports.deploymentsTotal,
      DEPLOYMENT_HISTORIES -> reports.deploymentHistoriesTotal,
      DEPLOYMENT_RECORDS -> reports.deploymentRecordsTotal,
      ITSM_RECORDS -> reports.itsmRecordsTotal,
      COMPLIANCE_RECORDS -> reports.complianceRecordsTotal,
      BUILD_RECORDS -> reports.buildRecordsTotal,
      PLAN_RECORDS -> reports.planRecordsTotal,
      ARCHIVED_DEPLOYMENT_RECORDS -> reports.archivedDeploymentRecordsTotal,
      ARCHIVED_ITSM_RECORDS -> reports.archivedItsmRecordsTotal,
      ARCHIVED_COMPLIANCE_RECORDS -> reports.archivedComplianceRecordsTotal,
      ARCHIVED_BUILD_RECORDS -> reports.archivedBuildRecordsTotal,
      ARCHIVED_PLAN_RECORDS -> reports.archivedPlanRecordsTotal,
      PERMISSION_SNAPSHOTS -> reports.permissionSnapshotsTotal,
      RISK_ASSESSMENTS -> reports.riskAssessmentsTotal
    ).toPendo

    PendoEvent(REPORT_METRICS, properties, Instant.now.toEpochMilli)
  }

  def generateUsageMetricsEvent(config: ConfigurationUsageStatistics, folder: FolderUsageStatistics, variable: VariableUsageStatistics,
                                dashboard: DashboardUsageStatistics, group: ReleaseGroupUsageStatistics, delivery: DeliveryUsageStatistics,
                                application: ApplicationUsageStatistics, environment: EnvironmentUsageStatistics, calendar: CalendarUsageStatistics):
  PendoEvent = {
    val properties = Map(
      "globalConfigurations" -> config.globalConfigurationsTotal,
      "folderConfigurations" -> config.folderConfigurationsTotal,
      FOLDERS -> folder.foldersTotal,
      MAX_FOLDER_DEPTH -> folder.maxDepthOfFolder,
      GLOBAL_VARIABLES -> variable.globalVariablesTotal,
      FOLDER_VARIABLES -> variable.folderVariablesTotal,
      AVG_VARIABLES_PER_FOLDER -> variable.averageVariablesPerFolder,
      GLOBAL_DASHBOARDS -> dashboard.globalDashboardsTotal,
      FOLDER_DASHBOARDS -> dashboard.folderDashboardsTotal,
      GROUPS -> group.groupsTotal,
      PATTERNS -> delivery.patternsTotal,
      DELIVERIES -> delivery.deliveriesTotal,
      APPLICATIONS -> application.applicationsTotal,
      ENVIRONMENTS -> environment.environmentsTotal,
      CALENDAR_ENTRIES -> calendar.calendarEntriesTotal
    ).toPendo

    PendoEvent(USAGE_METRICS, properties, Instant.now.toEpochMilli)
  }

  def generateArchiveMetricsEvent(archive: ArchiveUsageStatistics): PendoEvent = {
    val properties = Map(
      ARCHIVING_ENABLED -> archive.archivingEnabled,
      ARCHIVING_AGE -> archive.archivingAge,
      PRE_ARCHIVED_RELEASES -> archive.prearchivedReleasesTotal,
      ARCHIVED_RELEASES -> archive.archivedReleasesTotal,
      OLDEST_ARCHIVED_RELEASE_DATE -> archive.oldestArchivedRelease,
      ARCHIVED_COMPLETED_RELEASES -> archive.archivedCompletedReleasesTotal,
      ARCHIVED_ABORTED_RELEASES -> archive.archivedAbortedReleasesTotal,
      ARCHIVED_PHASES -> archive.archivedPhasesTotal,
      ARCHIVED_TASKS -> archive.archivedTasksTotal,
      ARCHIVED_ATTACHMENTS -> archive.archivedAttachmentsTotal,
      ARCHIVED_TAGS -> archive.archivedTagsTotal,
      ARCHIVED_ROLES -> archive.archivedRolesTotal,
      ARCHIVED_MEMBERS -> archive.archivedMembersTotal
    ).toPendo

    PendoEvent(ARCHIVE_METRICS, properties, Instant.now.toEpochMilli)
  }

  def generateLicenseMetricsEvent(license: LicenseUsageStatistics): PendoEvent = {
    val properties = Map(
      LICENSE_VERSION -> license.licenseVersion,
      LICENSED_PRODUCT -> license.licensedProduct,
      LICENSE_EDITION -> license.licensedEdition,
      LICENSED_TO -> license.licensedTo,
      LICENSE_EXPIRATION_DATE -> license.licenseExpiresAfter,
      LICENSE_REMAINING_DAYS -> license.licenseRemainingDays,
      LICENSE_UTILIZATION_RATE -> license.licenseUtilizationRate,
      LICENSE_REPOSITORY_ID -> license.licensedRepositoryId,
      LICENSE_MAX_USERS -> license.licensedMaxUsers,
      LICENSE_CURRENT_USERS -> license.currentUsers
    ).toPendo

    PendoEvent(LICENSE_METRICS, properties, Instant.now.toEpochMilli)
  }

  def generateInstallationMetricsEvent(general: GeneralUsageStatistics, cluster: ClusterUsageStatistics, installation: InstallationUsageStatistics): PendoEvent = {
    val properties = (Map(
      RELEASE_VERSION -> general.releaseVersion,
      INSTANCE_NAME -> general.instanceName,
      HOSTNAME -> general.hostname,
      SERVER_URL -> general.serverUrl,
      VM_NAME -> general.vmName,
      OS_NAME -> general.osName,
      PROCESSORS -> general.osAvailableProcessors,
      MAX_MEMORY -> general.maxMemory,
      USED_MEMORY -> general.usedMemory,
      MAIN_DB_TYPE -> general.databaseImplementation,
      ARCHIVE_DB_TYPE -> general.archiveDatabaseImplementation,
      NO_HIT_DB_QUERY_DURATION -> general.noHitDBQueryDuration,
      HIT_DB_QUERY_DURATION -> general.hitDBQueryDuration,
      INSTALLED_PLUGINS -> installation.officialPlugins.asScala.mkString(","),
      CUSTOM_PLUGINS -> installation.customPlugins.asScala.mkString(","),
      PLUGIN_COUNT -> installation.pluginsTotal,
      CUSTOM_PLUGIN_COUNT -> installation.customPluginsTotal,
      HOTFIXES -> installation.hotfixDirectory.asScala.mkString(","),
      EXTENSION_DIRECTORY -> installation.extDirectory.asScala.mkString(",")
    ) ++ createClusterMetrics(cluster) ++ createVmArgsMetrics(general)).toPendo

    PendoEvent(INSTALLATION_METRICS, properties, Instant.now.toEpochMilli)
  }

  private def createTop25TaskMetrics(top25Tasks: List[String]): Map[String, Any] = {
    val propertyArray = splitPropertyList(top25Tasks)
    Map(TOP_25_TASK_TYPES_1 -> propertyArray(0), TOP_25_TASK_TYPES_2 -> propertyArray(1))
  }

  private def createVmArgsMetrics(general: GeneralUsageStatistics): Map[String, Any] = {
    val propertyArray = splitPropertyList(general.vmArguments.split(",").toList)
    Map(VM_ARGS_1 -> propertyArray(0), VM_ARGS_2 -> propertyArray(1))
  }

  private def createClusterMetrics(cluster: ClusterUsageStatistics): Map[String, Any] = {
    if (0 == cluster.mode.compareToIgnoreCase(ClusterMode.Full.configOption)) {
      Map(
        CLUSTER_MODE -> cluster.mode,
        NUMBER_OF_NODES -> cluster.numberOfNodes,
        CLUSTER_LEADER -> cluster.leader,
        ACTIVE_NODE_DB -> cluster.seedNodes.asScala.mkString(","),
        ACTIVE_NODE_AKKA -> cluster.akkaNodes.asScala.mkString(",")
      )
    } else {
      Map(
        CLUSTER_MODE -> cluster.mode,
        NUMBER_OF_NODES -> "",
        CLUSTER_LEADER -> "",
        ACTIVE_NODE_DB -> "",
        ACTIVE_NODE_AKKA -> ""
      )
    }
  }
}
