package com.xebialabs.xlrelease.quartz.release.scheduler

import com.xebialabs.xlrelease.domain.PlanItem
import com.xebialabs.xlrelease.repository.Ids.getFolderlessId
import com.xebialabs.xlrelease.service.PendingPlanItemScheduledJobService
import grizzled.slf4j.Logging
import org.apache.commons.codec.digest.DigestUtils
import org.quartz.JobBuilder.newJob
import org.quartz.SimpleScheduleBuilder.simpleSchedule
import org.quartz.TriggerBuilder.newTrigger
import org.quartz.{JobDetail, Scheduler, TriggerKey}

import scala.jdk.CollectionConverters._

class QuartzPendingPlanItemScheduledJobService(scheduler: Scheduler, quartzJobs: List[PendingPlanItemQuartzJob[_]])
  extends PendingPlanItemScheduledJobService with Logging {

  import QuartzPendingPlanItemScheduledJobService._

  override def schedule[T <: PlanItem](planItem: T): Unit = {
    logger.debug(s"Schedule planItem [${planItem.getId}]")
    val jobClass = getApplicableQuartzJobClass(planItem)

    val job: JobDetail = newJob(jobClass)
      .withDescription(s"Job that will start pending planItem '${planItem.getName}' on ${planItem.getScheduledStartDate}")
      .withIdentity(planItem.hashedId, PENDING_PLAN_ITEM_TRIGGER_GROUP)
      .usingJobData(JOB_DATE_KEY, planItem.getId)
      .build()

    val quartzPlanItem = newTrigger()
      .withIdentity(planItem.hashedId, PENDING_PLAN_ITEM_TRIGGER_GROUP)
      .withDescription(s"PlanItem '${planItem.getName}'")
      .startAt(planItem.getScheduledStartDate)
      .withSchedule(simpleSchedule().withMisfireHandlingInstructionFireNow())
      .build()

    if (scheduler.checkExists(quartzPlanItem.getKey)) {
      scheduler.rescheduleJob(quartzPlanItem.getKey, quartzPlanItem)
    } else {
      scheduler.scheduleJob(job, Set(quartzPlanItem).asJava, true)
    }
  }

  override def unschedule(planItemId: String): Unit = {
    logger.debug(s"Un-schedule planItem [$planItemId]")
    val triggerKey = new TriggerKey(planItemId.hashedId, PENDING_PLAN_ITEM_TRIGGER_GROUP)
    scheduler.unscheduleJob(triggerKey)
  }

  private def getApplicableQuartzJobClass[T <: PlanItem](planItem: T): Class[_ <: PendingPlanItemQuartzJob[_]] = {
    quartzJobs.find(_.supports(planItem)).getOrElse(throw new IllegalArgumentException(s"Unsupported type '${planItem.getType}' for scheduler")).getClass
  }

}

object QuartzPendingPlanItemScheduledJobService {

  val PENDING_PLAN_ITEM_TRIGGER_GROUP: String = "PENDING_PLAN_ITEM"

  val JOB_DATE_KEY: String = "planItemId"

  implicit class PlanItemIdExtension[T <: PlanItem](planItem: T) {
    def hashedId: String = planItem.getId.hashedId
  }

  implicit class IdExtension(planItemId: String) {
    def hashedId: String = DigestUtils.sha256Hex(getFolderlessId(planItemId))
  }

}
