package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.configuration.ExecutionLogPurgeSettings
import com.xebialabs.xlrelease.domain.events.ConfigurationUpdatedEvent
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.quartz.config.QuartzConfiguration.BEAN_QUARTZ_SCHEDULER
import com.xebialabs.xlrelease.quartz.events.SchedulerStartedEvent
import com.xebialabs.xlrelease.service.TaskExecutionLogPurgeJob.LOG_PURGE_AGE_KEY
import grizzled.slf4j.Logging
import org.quartz.JobBuilder.newJob
import org.quartz.SimpleScheduleBuilder.simpleSchedule
import org.quartz.TriggerBuilder.newTrigger
import org.quartz.{JobDataMap, JobDetail, Scheduler, SimpleTrigger}
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.event
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
@EventListener
class TaskExecutionLogPurgeScheduler(xlrConfig: XlrConfig,
                                     @Qualifier(BEAN_QUARTZ_SCHEDULER) scheduler: Scheduler,
                                     configurationService: ConfigurationService)
  extends Logging {

  private val serviceName = "Task execution log purge"

  private val jobName = "TASK_EXECUTION_LOG_PURGE_JOB"
  private val groupName = "TASK_EXECUTION_LOG_PURGE"
  private val interval: Long = xlrConfig.jobLog.cleanupInterval

  lazy val job: JobDetail = newJob(classOf[TaskExecutionLogPurgeJob])
    .withDescription(jobName)
    .withIdentity(jobName, groupName)
    .storeDurably(true)
    .setJobData(
      new JobDataMap(
        Map[String, Int](LOG_PURGE_AGE_KEY -> getExecutionLogPurgeSettings().getExecutionLogsRetentionPeriod).asJava
      )
    )
    .build()

  lazy val trigger: SimpleTrigger = newTrigger()
    .withIdentity(jobName, groupName)
    .withDescription(jobName)
    .withSchedule(
      simpleSchedule().withIntervalInMilliseconds(interval).repeatForever()
        .withMisfireHandlingInstructionNowWithRemainingCount()
    )
    .startNow()
    .build()

  def resume(): Unit = {
    scheduler.resumeJob(job.getKey)
    logger.debug(s"Started $serviceName service")
  }

  def pause(): Unit = {
    scheduler.pauseJob(job.getKey)
    logger.debug(s"Stopped $serviceName service")
  }

  @event.EventListener
  def onStartup(event: SchedulerStartedEvent): Unit = {
    try {
      schedule()
    } catch {
      case e: Exception => logger.error(s"Error while starting $serviceName service", e)
    }
  }

  @AsyncSubscribe
  def onConfigurationUpdate(event: ConfigurationUpdatedEvent): Unit = {
    if (event.updated.getType.instanceOf(Type.valueOf(classOf[ExecutionLogPurgeSettings]))) {
      schedule()
    }
  }

  private def schedule(): Unit = {
    logger.trace(s"Scheduling $serviceName service")
    job.getJobDataMap.put(LOG_PURGE_AGE_KEY, getExecutionLogPurgeSettings().getExecutionLogsRetentionPeriod)
    scheduler.addJob(job, true)

    if (scheduler.checkExists(trigger.getKey)) {
      scheduler.rescheduleJob(trigger.getKey, trigger)
    } else {
      scheduler.scheduleJob(job, Set(trigger).asJava, true)
    }
  }

  private def getExecutionLogPurgeSettings(): ExecutionLogPurgeSettings = {
    configurationService.read(ExecutionLogPurgeSettings.EXECUTION_LOG_SETTINGS_ID).asInstanceOf[ExecutionLogPurgeSettings]
  }
}
