package com.xebialabs.xlrelease.quartz.config

import com.xebialabs.xlplatform.cluster.ClusterMode.Standalone
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.db.sql.DatabaseInfo.{Db2, MsSqlServer, Oracle, PostgreSql}
import com.xebialabs.xlrelease.quartz.config.QuartzConfiguration._
import com.xebialabs.xlrelease.spring.config.SqlConfiguration
import grizzled.slf4j.{Logger, Logging}
import org.quartz.Scheduler
import org.quartz.impl.StdSchedulerFactory._
import org.quartz.impl.jdbcjobstore.oracle.OracleDelegate
import org.quartz.impl.jdbcjobstore.{DB2v8Delegate, MSSQLDelegate, PostgreSQLDelegate, StdJDBCDelegate}
import org.quartz.spi.JobFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.config.PropertiesFactoryBean
import org.springframework.context.ApplicationEventPublisher
import org.springframework.context.annotation.{Bean, Configuration}
import org.springframework.core.io.ClassPathResource
import org.springframework.scheduling.quartz.{LocalDataSourceJobStore, SchedulerFactoryBean, SpringBeanJobFactory}

import java.io.IOException
import java.util.Properties
import java.util.concurrent.Executor

@Configuration
class QuartzConfiguration(sqlConfiguration: SqlConfiguration, xlrConfig: XlrConfig) extends Logging {

  @transient override lazy val logger = Logger(classOf[QuartzConfiguration])

  @Autowired
  var eventPublisher: ApplicationEventPublisher = _

  // Scheduler to run release triggers
  @Bean(name = Array(BEAN_TRIGGER_SCHEDULER))
  def triggerScheduler(): Scheduler = {
    triggerSchedulerFactory().getObject
  }

  // Scheduler to run any quartz job except release triggers
  @Bean(name = Array(BEAN_QUARTZ_JOB_SCHEDULER))
  def quartzJobScheduler(): Scheduler = {
    quartzJobSchedulerFactory().getObject
  }

  @Bean
  def quartzInitializer(): QuartzInitializer = new QuartzInitializer(List(triggerScheduler(), quartzJobScheduler()), eventPublisher)

  @Bean
  def triggerSchedulerFactory(): SchedulerFactoryBean = {
    createSchedulerFactory(TRIGGER_SCHEDULER_NAME, xlrConfig.executors.releaseTrigger.pool)
  }

  @Bean
  def quartzJobSchedulerFactory(): SchedulerFactoryBean = {
    createSchedulerFactory(QUARTZ_JOB_SCHEDULER_NAME, xlrConfig.executors.quartzJobExecutor.pool)
  }

  @Bean
  def jobFactory(): JobFactory = {
    new SpringBeanJobFactory()
  }

  @Bean
  def quartzProperties(): Properties = {
    val propertiesFactoryBean = new PropertiesFactoryBean
    propertiesFactoryBean.setLocation(new ClassPathResource("/quartz.properties"))
    val props = try {
      propertiesFactoryBean.afterPropertiesSet()
      new Properties(propertiesFactoryBean.getObject)
    } catch {
      case e: IOException =>
        logger.debug("Cannot load quartz.properties.")
        new Properties()
    }
    configureJobStore(props)
    configureClustering(props)
    props
  }

  private def createSchedulerFactory(schedulerName: String, taskExecutor: Executor): SchedulerFactoryBean = {
    val quartzScheduler = new SchedulerFactoryBean()
    quartzScheduler.setDataSource(sqlConfiguration.xlrRepositoryDataSourceProxy())
    quartzScheduler.setTransactionManager(sqlConfiguration.xlrRepositoryTransactionManager())
    quartzScheduler.setOverwriteExistingJobs(true)
    quartzScheduler.setAutoStartup(false)
    quartzScheduler.setSchedulerName(schedulerName)
    quartzScheduler.setTaskExecutor(taskExecutor)
    quartzScheduler.setJobFactory(jobFactory())
    quartzScheduler.setQuartzProperties(quartzProperties())

    quartzScheduler
  }

  private def configureJobStore(props: Properties): Unit = {
    props.put(s"$PROP_JOB_STORE_USE_PROP", java.lang.Boolean.TRUE)
    props.put(s"$PROP_JOB_STORE_CLASS", classOf[LocalDataSourceJobStore].getCanonicalName)
    props.put(s"$PROP_JOB_STORE_PREFIX.$PROP_TABLE_PREFIX", QUARTZ_TABLE_PREFIX)
    val driverDelegateClass = sqlConfiguration.xlrDbInfo() match {
      case Db2(_) => classOf[DB2v8Delegate].getCanonicalName
      case Oracle(_) => classOf[OracleDelegate].getCanonicalName
      case MsSqlServer(_) => classOf[MSSQLDelegate].getCanonicalName
      case PostgreSql(_) => classOf[PostgreSQLDelegate].getCanonicalName
      case _ => classOf[StdJDBCDelegate].getCanonicalName
    }
    props.put(s"$PROP_JOB_STORE_PREFIX.driverDelegateClass", driverDelegateClass)
  }

  private def configureClustering(props: Properties): Unit = {
    val FALSE = Boolean.box(false)
    val isClustered = xlrConfig.cluster.mode != Standalone
    props.put(s"$PROP_JOB_STORE_PREFIX.isClustered", Boolean.box(isClustered))
    if (isClustered) {
      /*
       * IMPORTANT: Never run clustering on separate machines,
       * unless their clocks are synchronized using some form
       * of time-sync service (daemon) that runs very regularly
       * (the clocks must be within a second of each other).
       *
       * See http://www.boulder.nist.gov/timefreq/service/its.htm
       * if you are unfamiliar with how to do this.
       *
       * Never start (scheduler.start()) a non-clustered instance
       * against the same set of database tables that any other
       * instance is running (start()ed) against.
       *
       * You may get serious data corruption, and will definitely
       * experience erratic behavior.
       */
      props.put(PROP_SCHED_INSTANCE_ID, "AUTO")
      props.put("org.quartz.jobStore.misfireThreshold", "60000")
      props.put("org.quartz.jobStore.clusterCheckinInterval", "20000")
    }
    props.put(PROP_SCHED_RMI_EXPORT, FALSE)
    props.put(PROP_SCHED_RMI_PROXY, FALSE)
  }

}

object QuartzConfiguration {
  final val BEAN_TRIGGER_SCHEDULER = "xlrTriggerScheduler"
  final val TRIGGER_SCHEDULER_NAME = "xlrTriggerScheduler"
  final val BEAN_QUARTZ_JOB_SCHEDULER = "xlrQuartzJobScheduler"
  final val QUARTZ_JOB_SCHEDULER_NAME = "xlrQuartzJobScheduler"
  final val QUARTZ_TABLE_PREFIX = "Q_"
}
