package com.xebialabs.xlrelease.quartz.release.scheduler

import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.quartz.config.QuartzConfiguration.QUARTZ_SCHEDULER_NAME
import com.xebialabs.xlrelease.quartz.events.SchedulerStartedEvent
import com.xebialabs.xlrelease.service.FeatureService
import grizzled.slf4j.Logging
import org.quartz.{JobDetail, Trigger}
import org.springframework.context.event

@IsTransactional
trait QuartzScheduledService extends FeatureService with Logging {

  private var running: Boolean = false

  def releaseSchedulerService: ReleaseSchedulerService

  def getJobDetail: JobDetail

  def getTrigger: Trigger

  @event.EventListener
  def onStartup(event: SchedulerStartedEvent): Unit = {
    if (null != event && null != event.schedulerName && event.schedulerName == QUARTZ_SCHEDULER_NAME) {
      try {
        if (isEnabled) {
          logger.debug(s"Scheduling ${name()} service")
          releaseSchedulerService.schedule(getJobDetail, getTrigger)
          running = true
        } else {
          logger.debug(s"Removing ${name()} service")
          releaseSchedulerService.unschedule(getJobDetail.getKey)
          running = false
        }
      } catch {
        case e: Exception => logger.error(s"Error while starting ${name()} service", e)
      }
    }
  }

  def resume(): Unit = {
    if (!running) {
      releaseSchedulerService.resume(getJobDetail.getKey)
      logger.debug(s"Started ${name()} service")
    }
    running = true
  }

  def pause(): Unit = {
    if (running) {
      releaseSchedulerService.pause(getJobDetail.getKey)
      logger.debug(s"Stopped ${name()} service")
    }
    running = false
  }

  override def enable(): Unit = resume()

  override def disable(): Unit = pause()

  override def stop(): Unit = {
    running = false
    releaseSchedulerService.getScheduler.shutdown()
  }
}
