package com.xebialabs.xlrelease.quartz.release.scheduler

import grizzled.slf4j.Logging
import org.quartz.impl.matchers.GroupMatcher
import org.springframework.jmx.`export`.annotation.{ManagedOperation, ManagedResource}
import org.springframework.stereotype.Component

import java.util

@Component
@ManagedResource(objectName = "com.xebialabs.xlrelease.settings:name=ScheduledServicesControl", description = "Control scheduled services")
class ScheduledServiceControl(overdueNotificationService: OverdueNotificationService,
                              dueSoonNotificationService: DueSoonNotificationService,
                              releaseSchedulerService: ReleaseSchedulerService) extends Logging {

  @ManagedOperation(description = "Pause all Quartz triggers")
  def pauseQuartz(): Unit = {
    releaseSchedulerService.getScheduler.pauseAll()
    logger.info("Paused Quartz triggers")
  }

  @ManagedOperation(description = "Resume (un-pause) all Quartz triggers")
  def resumeQuartz(): Unit = {
    releaseSchedulerService.getScheduler.resumeAll()
    logger.info("Resumed Quartz triggers")
  }

  @ManagedOperation(description = "List Quartz trigger groups")
  def getTriggerGroupNames(): util.List[String] = {
    releaseSchedulerService.getScheduler.getTriggerGroupNames
  }

  @ManagedOperation(description = "Pause Quartz trigger group")
  def pauseTriggerGroup(triggerGroupName: String): Unit = {
    releaseSchedulerService.getScheduler.pauseTriggers(GroupMatcher.groupEquals(triggerGroupName))
    logger.info(s"Paused Quartz trigger group $triggerGroupName")
  }

  @ManagedOperation(description = "Resume Quartz trigger group")
  def resumeTriggerGroup(triggerGroupName: String): Unit = {
    releaseSchedulerService.getScheduler.resumeTriggers(GroupMatcher.groupEquals(triggerGroupName))
    logger.info(s"Resumed Quartz trigger group $triggerGroupName")
  }

  @ManagedOperation(description = "Pause overdue notifications")
  def pauseOverdueNotification(): Unit = {
    overdueNotificationService.pause()
  }

  @ManagedOperation(description = "Resume (un-pause) overdue notifications")
  def resumeOverdueNotification(): Unit = {
    overdueNotificationService.resume()
  }

  @ManagedOperation(description = "Pause due-soon notifications")
  def pauseDueSoonNotification(): Unit = {
    dueSoonNotificationService.pause()
  }

  @ManagedOperation(description = "Resume (un-pause) due-soon notifications")
  def resumeDueSoonNotification(): Unit = {
    dueSoonNotificationService.resume()
  }

}
