package com.xebialabs.xlrelease.quartz.config

import com.xebialabs.xlplatform.cluster.ClusterMode.Standalone
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.db.sql.DatabaseInfo.{Db2, MsSqlServer, Oracle, PostgreSql}
import com.xebialabs.xlrelease.quartz.config.QuartzConfiguration._
import com.xebialabs.xlrelease.quartz.release.scheduler.ReleaseSchedulerService
import com.xebialabs.xlrelease.scheduler.RestartableExecutorService
import com.xebialabs.xlrelease.service.QuartzLifecycleManager
import com.xebialabs.xlrelease.spring.config.SqlConfiguration
import grizzled.slf4j.{Logger, Logging}
import org.quartz.impl.StdSchedulerFactory._
import org.quartz.impl.jdbcjobstore.oracle.OracleDelegate
import org.quartz.impl.jdbcjobstore.{DB2v8Delegate, MSSQLDelegate, PostgreSQLDelegate, StdJDBCDelegate}
import org.quartz.spi.JobFactory
import org.springframework.beans.factory.config.PropertiesFactoryBean
import org.springframework.context.ApplicationEventPublisher
import org.springframework.context.annotation.{Bean, Configuration, Primary}
import org.springframework.core.io.ClassPathResource
import org.springframework.scheduling.quartz.{LocalDataSourceJobStore, SchedulerFactoryBean, SpringBeanJobFactory}

import java.io.IOException
import java.util.Properties
import java.util.concurrent.Executor

@Configuration
class QuartzConfiguration(sqlConfiguration: SqlConfiguration, xlrConfig: XlrConfig, eventPublisher: ApplicationEventPublisher) extends Logging {

  @transient override lazy val logger = Logger(classOf[QuartzConfiguration])

  @Bean
  def triggerSchedulerService(): ReleaseSchedulerService = {
    new ReleaseSchedulerService(triggerSchedulerFactory())
  }

  // Scheduler to run release triggers
  @Bean
  def triggerSchedulerFactory(): SchedulerFactoryBean = {
    createSchedulerFactory(TRIGGER_SCHEDULER_NAME, xlrConfig.executors.releaseTrigger.pool)
  }

  @Bean
  @Primary
  def quartzJobSchedulerService(): ReleaseSchedulerService = {
    new ReleaseSchedulerService(quartzJobSchedulerFactory())
  }

  // Scheduler to run any quartz job except release triggers
  @Bean
  @Primary
  def quartzJobSchedulerFactory(): SchedulerFactoryBean = {
    createSchedulerFactory(QUARTZ_SCHEDULER_NAME, xlrConfig.executors.quartzJobExecutor.pool)
  }

  @Bean
  def quartzLifecycleManager(): QuartzLifecycleManager = {
    val schedulers = List(triggerSchedulerFactory(), quartzJobSchedulerFactory());
    val executors = List(
      xlrConfig.executors.releaseTrigger.pool.asInstanceOf[RestartableExecutorService],
      xlrConfig.executors.quartzJobExecutor.pool.asInstanceOf[RestartableExecutorService]
    )
    new QuartzLifecycleManager(schedulers, executors, eventPublisher)
  }

  @Bean
  def jobFactory(): JobFactory = {
    new SpringBeanJobFactory()
  }

  @Bean
  def quartzProperties(): Properties = {
    val propertiesFactoryBean = new PropertiesFactoryBean
    propertiesFactoryBean.setLocation(new ClassPathResource("/quartz.properties"))
    val props = try {
      propertiesFactoryBean.afterPropertiesSet()
      new Properties(propertiesFactoryBean.getObject)
    } catch {
      case e: IOException =>
        logger.debug("Cannot load quartz.properties.")
        new Properties()
    }
    configureJobStore(props)
    configureClustering(props)
    validateQuartzProperties(props)

    props
  }

  private def validateQuartzProperties(props: Properties): Unit = {
    props.forEach((key, value) => {
      if (!value.isInstanceOf[String]) {
        throw new IllegalStateException(s"Quartz property '$key' value should be string")
      }
    })
  }

  private def createSchedulerFactory(schedulerName: String, taskExecutor: Executor): SchedulerFactoryBean = {
    val quartzScheduler = new SchedulerFactoryBean()
    quartzScheduler.setDataSource(sqlConfiguration.xlrRepositoryDataSourceProxy())
    quartzScheduler.setTransactionManager(sqlConfiguration.xlrRepositoryTransactionManager())
    quartzScheduler.setOverwriteExistingJobs(true)
    quartzScheduler.setAutoStartup(false)
    quartzScheduler.setSchedulerName(schedulerName)
    quartzScheduler.setTaskExecutor(taskExecutor)
    quartzScheduler.setJobFactory(jobFactory())
    quartzScheduler.setWaitForJobsToCompleteOnShutdown(true)
    quartzScheduler.setQuartzProperties(quartzProperties())

    quartzScheduler
  }

  private def configureJobStore(props: Properties): Unit = {
    props.put(s"$PROP_JOB_STORE_USE_PROP", java.lang.Boolean.TRUE.toString)
    props.put(s"$PROP_JOB_STORE_CLASS", classOf[LocalDataSourceJobStore].getCanonicalName)
    props.put(s"$PROP_JOB_STORE_PREFIX.$PROP_TABLE_PREFIX", QUARTZ_TABLE_PREFIX)
    val driverDelegateClass = sqlConfiguration.xlrDbInfo() match {
      case Db2(_) => classOf[DB2v8Delegate].getCanonicalName
      case Oracle(_) => classOf[OracleDelegate].getCanonicalName
      case MsSqlServer(_) =>
        props.put(s"$PROP_JOB_STORE_PREFIX.acquireTriggersWithinLock", java.lang.Boolean.TRUE.toString)
        classOf[MSSQLDelegate].getCanonicalName
      case PostgreSql(_) => classOf[PostgreSQLDelegate].getCanonicalName
      case _ => classOf[StdJDBCDelegate].getCanonicalName
    }
    props.put(s"$PROP_JOB_STORE_PREFIX.driverDelegateClass", driverDelegateClass)
  }

  private def configureClustering(props: Properties): Unit = {
    val FALSE = Boolean.box(false)
    val isClustered = xlrConfig.cluster.mode != Standalone
    props.put(s"$PROP_JOB_STORE_PREFIX.isClustered", Boolean.box(isClustered).toString)
    if (isClustered) {
      /*
       * IMPORTANT: Never run clustering on separate machines,
       * unless their clocks are synchronized using some form
       * of time-sync service (daemon) that runs very regularly
       * (the clocks must be within a second of each other).
       *
       * See http://www.boulder.nist.gov/timefreq/service/its.htm
       * if you are unfamiliar with how to do this.
       *
       * Never start (scheduler.start()) a non-clustered instance
       * against the same set of database tables that any other
       * instance is running (start()ed) against.
       *
       * You may get serious data corruption, and will definitely
       * experience erratic behavior.
       */
      props.put(PROP_SCHED_INSTANCE_ID, AUTO_GENERATE_INSTANCE_ID)
      props.put("org.quartz.jobStore.misfireThreshold", "60000")
      props.put("org.quartz.jobStore.clusterCheckinInterval", "20000")
    }
    props.put(PROP_SCHED_RMI_EXPORT, FALSE.toString)
    props.put(PROP_SCHED_RMI_PROXY, FALSE.toString)
  }

}

object QuartzConfiguration {
  final val TRIGGER_SCHEDULER_NAME = "xlrTriggerScheduler"
  final val QUARTZ_SCHEDULER_NAME = "xlrQuartzScheduler"
  final val QUARTZ_TABLE_PREFIX = "Q_"
}
