package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.configuration.ExecutionLogPurgeSettings
import com.xebialabs.xlrelease.domain.events.ConfigurationUpdatedEvent
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.quartz.release.scheduler.{QuartzScheduledService, ReleaseSchedulerService}
import com.xebialabs.xlrelease.service.TaskExecutionLogPurgeJob.LOG_PURGE_AGE_KEY
import org.quartz.JobBuilder.newJob
import org.quartz.SimpleScheduleBuilder.simpleSchedule
import org.quartz.TriggerBuilder.newTrigger
import org.quartz.{JobDataMap, JobDetail, SimpleTrigger, Trigger}
import org.springframework.stereotype.Component

import java.util.Date
import scala.jdk.CollectionConverters._

@Component
@EventListener
class TaskExecutionLogPurgeScheduler(xlrConfig: XlrConfig,
                                     val releaseSchedulerService: ReleaseSchedulerService,
                                     configurationService: ConfigurationService)
  extends QuartzScheduledService {

  private val serviceName = "Task execution log purge"

  private val jobName = "TASK_EXECUTION_LOG_PURGE_JOB"
  private val groupName = "TASK_EXECUTION_LOG_PURGE"
  private val interval: Long = xlrConfig.jobLog.cleanupInterval
  private val delay: Long = xlrConfig.durations.startupJobsDelay.toMillis


  lazy val job: JobDetail = newJob(classOf[TaskExecutionLogPurgeJob])
    .withDescription(jobName)
    .withIdentity(jobName, groupName)
    .storeDurably(true)
    .setJobData(
      new JobDataMap(
        Map[String, String](LOG_PURGE_AGE_KEY -> getExecutionLogPurgeSettings().getExecutionLogsRetentionPeriod.toString).asJava
      )
    )
    .build()

  lazy val trigger: SimpleTrigger = newTrigger()
    .withIdentity(jobName, groupName)
    .withDescription(jobName)
    .withSchedule(
      simpleSchedule().withIntervalInMilliseconds(interval).repeatForever()
        .withMisfireHandlingInstructionNextWithRemainingCount()
    )
    .startAt(Date.from(new Date().toInstant.plusMillis(delay)))
    .build()

  @AsyncSubscribe
  def onConfigurationUpdate(event: ConfigurationUpdatedEvent): Unit = {
    if (event.updated.getType.instanceOf(Type.valueOf(classOf[ExecutionLogPurgeSettings]))) {
      logger.trace(s"Scheduling $serviceName service")
      job.getJobDataMap.put(LOG_PURGE_AGE_KEY, getExecutionLogPurgeSettings().getExecutionLogsRetentionPeriod.toString)
      releaseSchedulerService.schedule(job, trigger)
    }
  }

  private def getExecutionLogPurgeSettings(): ExecutionLogPurgeSettings = {
    configurationService.read(ExecutionLogPurgeSettings.EXECUTION_LOG_SETTINGS_ID).asInstanceOf[ExecutionLogPurgeSettings]
  }

  override def getJobDetail: JobDetail = job

  override def getTrigger: Trigger = trigger

  override def name(): String = serviceName

  override def isEnabled: Boolean = true
}
