package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.management.ServiceState._
import com.xebialabs.xlrelease.domain.management.{CompositeServiceStatus, ServiceStatus, SimpleServiceStatus}
import com.xebialabs.xlrelease.quartz.events.{SchedulerStartedEvent, SchedulerStartingEvent}
import com.xebialabs.xlrelease.scheduler.RestartableExecutorService
import grizzled.slf4j.Logging
import org.springframework.context.ApplicationEventPublisher
import org.springframework.scheduling.quartz.SchedulerFactoryBean

class QuartzLifecycleManager(schedulers: List[SchedulerFactoryBean],
                             executors: List[RestartableExecutorService],
                             eventPublisher: ApplicationEventPublisher,
                             xlrConfig: XlrConfig)
  extends XlrServiceLifecycle with Logging with FeatureService {


  override def getOrder(): Int = XlrServiceLifecycleOrder.QUARTZ

  override def doStart(): Unit = {
    if (xlrConfig.maintenanceModeEnabled) {
      // Quartz scheduling is disabled so job recovery will also be disabled
      logger.warn("[MAINTENANCE MODE] Scheduled releases and tasks - DISABLED")
      logger.warn("[MAINTENANCE MODE] Polling triggers - DISABLED")
    } else {
      executors.foreach(_.start())
      schedulers.foreach { scheduler =>
        val schedulerName = scheduler.getScheduler.getSchedulerName
        try {
          logger.debug(s"Starting quartz scheduler '$schedulerName'")
          eventPublisher.publishEvent(SchedulerStartingEvent(this, schedulerName))
          scheduler.start()
          logger.debug(s"Started quartz scheduler '$schedulerName'")
          eventPublisher.publishEvent(SchedulerStartedEvent(this, schedulerName))
        } catch {
          case e: Exception => logger.error(s"Error while starting quartz scheduler $schedulerName", e)
        }
      }
    }
  }

  override def doStop(): Unit = {
    schedulers.foreach(scheduler => {
      logger.debug(s"Stopping quartz scheduler ${scheduler.getScheduler.getSchedulerName}")
      scheduler.stop()
    })
    //This will actually stop already scheduled jobs and block until it completes.
    executors.foreach(_.stop())
  }


  override def serviceStatus(): ServiceStatus = {
    val statuses: List[SimpleServiceStatus] = schedulerStatuses
    val computedState = ServiceStatus.computeState(state, statuses)
    CompositeServiceStatus(serviceName(), computedState, statuses)
  }

  private def schedulerStatuses: List[SimpleServiceStatus] = {
    val statuses = schedulers.map(s => {
      val activeJobs = s.getScheduler.getCurrentlyExecutingJobs.size()
      val status = s.isRunning match {
        case true => Running
        case false if activeJobs > 0 => Stopping
        case false => Stopped
      }
      SimpleServiceStatus(s.getScheduler.getSchedulerName, status)
    })
    statuses
  }
}
