package com.xebialabs.xlrelease.scheduler

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.quartz.config.QuartzConfiguration.QUARTZ_SCHEDULER_NAME
import com.xebialabs.xlrelease.quartz.events.SchedulerStartingEvent
import com.xebialabs.xlrelease.quartz.release.scheduler.ReleaseSchedulerService
import grizzled.slf4j.Logging
import org.quartz.JobBuilder.newJob
import org.quartz.SimpleScheduleBuilder.simpleSchedule
import org.quartz.TriggerBuilder.newTrigger
import org.quartz.{JobDetail, Trigger}
import org.springframework.context.event
import org.springframework.stereotype.Component

import java.util.Date

@Component
class ScheduledTaskRecovery(xlrConfig: XlrConfig, releaseSchedulerService: ReleaseSchedulerService) extends Logging {

  val JOB_NAME = "SCHEDULED_TASK_RECOVERY_JOB"
  val GROUP_NAME = "SCHEDULED_TASK_RECOVERY"

  private val delay: Long = xlrConfig.durations.startupJobsDelay.toMillis

  @event.EventListener
  def onStartup(event: SchedulerStartingEvent): Unit = {
    if (null != event && null != event.schedulerName && event.schedulerName == QUARTZ_SCHEDULER_NAME) {
      try {
        val job: JobDetail = newJob(classOf[ScheduledTaskRecoveryJob])
          .withDescription(JOB_NAME)
          .withIdentity(JOB_NAME, GROUP_NAME)
          .storeDurably(true)
          .build()
        val trigger: Trigger = newTrigger()
          .withIdentity(JOB_NAME, GROUP_NAME)
          .withDescription(JOB_NAME)
          .withSchedule(simpleSchedule()
            .withIntervalInMilliseconds(XlrConfig.getInstance.timeouts.nodeFailureDetectionPeriod.toMillis)
            .repeatForever()
            .withMisfireHandlingInstructionNextWithRemainingCount())
          .startAt(Date.from(new Date().toInstant.plusMillis(delay)))
          .build()
        releaseSchedulerService.schedule(job, trigger)
      } catch {
        case e: Exception => logger.error(s"Error while starting ${classOf[ScheduledTaskRecovery].getName} service", e)
      }
    }
  }

}
