package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.security.PermissionEnforcer
import com.xebialabs.xlrelease.api.internal.EffectiveSecurityDecorator.EFFECTIVE_SECURITY
import com.xebialabs.xlrelease.api.internal.{DecoratorsCache, InternalMetadataDecoratorService}
import com.xebialabs.xlrelease.api.v1.forms.{ReleaseGroupFilters, ReleaseGroupOrderMode}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.group.ReleaseGroup
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence._
import com.xebialabs.xlrelease.repository.{Page, ReleaseGroupRepository}
import com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_RELEASE_GROUP
import com.xebialabs.xlrelease.utils.Diff
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Repository

import java.util
import scala.jdk.CollectionConverters._

@IsTransactional
@Repository
class SqlReleaseGroupRepository(releaseGroupPersistence: ReleaseGroupPersistence,
                                releaseGroupMemberPersistence: ReleaseGroupMemberPersistence,
                                sqlRepositoryAdapter: SqlRepositoryAdapter,
                                implicit val permissionEnforcer: PermissionEnforcer,
                                @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                val decoratorService: InternalMetadataDecoratorService) extends ReleaseGroupRepository {

  @IsReadOnly
  override def read(groupId: String): ReleaseGroup = releaseGroupPersistence.findById(groupId)
    .map(row => toReleaseGroup(row, releaseGroupMemberPersistence.findMembersByGroupUid(row.ciUid)))
    .map(releaseGroup => decorateWithEffectiveSecurity(Seq(releaseGroup), new DecoratorsCache()).head)
    .getOrElse(throw new LogFriendlyNotFoundException(s"Release group $groupId not found"))

  override def create(releaseGroup: ReleaseGroup): Unit = {
    val groupUid = releaseGroupPersistence.insert(releaseGroup)
    releaseGroupMemberPersistence.insertMembers(groupUid, releaseGroup.getReleaseIds.asScala.toSet)
  }

  override def update(releaseGroup: ReleaseGroup): Unit = {
    releaseGroupPersistence.update(releaseGroup)
    val groupUid = releaseGroupPersistence.getUid(releaseGroup.getId).get

    val existingMembers = releaseGroupMemberPersistence.findMembersByGroupUid(groupUid).map(getName)
    val currentMembers = releaseGroup.getReleaseIds.asScala.toSet.map(getName)
    val membersDiff = Diff(existingMembers, currentMembers)

    releaseGroupMemberPersistence.deleteMembers(groupUid, membersDiff.deletedKeys)
    releaseGroupMemberPersistence.insertMembers(groupUid, membersDiff.newKeys)
  }

  @IsReadOnly
  override def exists(groupId: String): Boolean = releaseGroupPersistence.exists(groupId)

  override def delete(groupId: String): Unit = releaseGroupPersistence.delete(groupId)

  @IsReadOnly
  override def search(filters: ReleaseGroupFilters, page: Page, orderBy: ReleaseGroupOrderMode,
                      principals: Iterable[String], roleIds: Iterable[String]): Seq[ReleaseGroup] = {
    val groupRows = if (filters.hasStatuses) {
      val sqlWithParams = new ReleaseGroupSqlBuilder()
        .select()
        .withTitleLike(filters.getTitle)
        .withOneOfStatuses(Option(filters.getStatuses).map(_.asScala.toSeq).getOrElse(Seq.empty))
        .withFolder(filters.getFolderId)
        .withPermission(if (permissionEnforcer.isCurrentUserAdmin) None else Some(VIEW_RELEASE_GROUP), principals, roleIds)
        .orderBy(orderBy)
        .limitAndOffset(page.resultsPerPage, page.resultsPerPage * page.page)
        .build()
      releaseGroupPersistence.findByQuery(sqlWithParams)
    } else {
      Seq.empty
    }

    val releaseGroups = toReleaseGroups(groupRows)
    if (permissionEnforcer.isCurrentUserAdmin) releaseGroups else decorateWithEffectiveSecurity(releaseGroups, new DecoratorsCache())
  }

  override def findGroupsReferencingReleaseId(releaseId: String): Seq[String] = releaseGroupMemberPersistence.findGroupsReferencingReleaseId(releaseId)

  @IsReadOnly
  override def findActiveGroupsReferencingFolderId(folderId: String): Seq[CiIdWithTitle] =
    releaseGroupMemberPersistence.findActiveGroupsReferencingFolderId(folderId)

  private def toReleaseGroups(rows: Seq[ReleaseGroupRow]): Seq[ReleaseGroup] = {
    val releaseIdsByGroup = releaseGroupMemberPersistence.findMembersByGroupUids(rows.map(_.ciUid))
    rows.map(row => toReleaseGroup(row, releaseIdsByGroup.getOrElse(row.ciUid, Seq.empty)))
  }

  private def toReleaseGroup(row: ReleaseGroupRow, newReleaseIds: Seq[String]): ReleaseGroup = {
    val releaseGroupOption = sqlRepositoryAdapter.deserialize[ReleaseGroup](row.content)
    releaseGroupOption match {
      case Some(releaseGroup) =>
        releaseGroup.setFolderId(row.folderId.absolute)
        val mergedReleaseIds = releaseGroup.getReleaseIds.asScala.map { oldId =>
          newReleaseIds.find(newId => getName(newId) == getName(oldId)).getOrElse(oldId)
        }
        releaseGroup.setReleaseIds(new util.HashSet(mergedReleaseIds.asJava))
        releaseGroup
      case None =>
        throw new LogFriendlyNotFoundException(s"Error reading release group ${row.ciUid}, see logs for more details")
    }

  }

  @IsReadOnly
  override def findFolderId(groupId: String): String = releaseGroupPersistence.findFolderId(groupId).absolute

  private def decorateWithEffectiveSecurity(releaseGroups: Seq[ReleaseGroup], cache: DecoratorsCache): Seq[ReleaseGroup] = releaseGroups.map(group => {
    decoratorService.decorate(group, Seq(EFFECTIVE_SECURITY).asJava, cache)
    group
  })
}
