package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.v1.views.{PhaseTimeline, ReleaseGroupTimeline, ReleaseTimeline}
import com.xebialabs.xlrelease.domain.group.ReleaseGroup
import com.xebialabs.xlrelease.domain.{Dependency, Phase, PlanItem, Release}
import com.xebialabs.xlrelease.planner.Plans.{Computed, Container}
import com.xebialabs.xlrelease.planner.{Planner, PlannerReleaseItem, PlannerReleaseTree}
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.repository.ReleaseRepository
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import grizzled.slf4j.Logging
import org.joda.time.DateTime

import java.util.Date
import scala.annotation.tailrec
import scala.jdk.CollectionConverters._

trait ReleaseGroupTimelineCalculator extends Logging {
  val releaseService: ReleaseService
  val releaseRepository: ReleaseRepository

  def calculateTimeline(releaseGroup: ReleaseGroup, now: DateTime): ReleaseGroupTimeline = {
    var dependentReleases = Set.empty[Release]
    val releaseTimelines = getReleasesOf(releaseGroup).flatMap { release =>
      dependentReleases = if (release.isArchived || dependentReleases.contains(release)) dependentReleases else dependentReleasesOf(release, dependentReleases)
      val tree = PlannerReleaseTree(release, dependentReleases.map(PlannerReleaseItem.transform).toList)
      Planner.makePlan(tree, now, usesScheduledDates).flatMap(toReleaseTimeline)
    }

    val groupTimeline = new ReleaseGroupTimeline(releaseGroup)
    groupTimeline.setReleases(releaseTimelines.asJava)
    groupTimeline
  }

  private def getReleasesOf(releaseGroup: ReleaseGroup): Seq[Release] = {
    releaseGroup.getReleaseIds.asScala.flatMap { releaseId =>
      try {
        Option(releaseService.findByIdIncludingArchived(releaseId))
      } catch {
        case e: NotFoundException => logger.warn(e.getMessage)
          None
      }
    }.toSeq
  }

  private def usesScheduledDates(item: PlannerReleaseItem) = !item.isPhase && !item.isRelease

  private def dependentReleasesOf(release: Release, collectedReleases: Set[Release]): Set[Release] = {
    @tailrec
    def gather0(toCollectFrom: Set[Release], collected: Set[Release]): Set[Release] = {
      toCollectFrom.toList match {
        case Nil => collected
        case head :: tail =>
          val releaseIds = head.getAllGates.asScala.flatMap { gate =>
            gate.getDependencies.asScala.flatMap {
              case dep: Dependency if dep.hasResolvedTarget =>
                Option(dep.getTarget[PlanItem]).map(target => releaseIdFrom(target.getId))
              case _ => None
            }
          }
          val newReleases = releaseIds.filterNot { releaseId =>
            collected.exists(_.getId == releaseId) || toCollectFrom.exists(_.getId == releaseId)
          }.map {
            releaseRepository.findById(_, ResolveOptions.WITHOUT_DECORATORS)
          }
          gather0((newReleases ++ tail).toSet, collected + head)
      }
    }

    gather0(Set(release), collectedReleases) - release
  }

  private def toReleaseTimeline(plan: Computed): Option[ReleaseTimeline] = plan match {
    case releasePlan: Container =>
      val phaseTimelines = releasePlan.children.sortBy(_.start).flatMap(toPhaseTimeline)
      val releaseTimeline = new ReleaseTimeline(
        releasePlan.item.item.asInstanceOf[Release], releasePlan.start.toDate, getReleaseEndDate(releasePlan, phaseTimelines)
      )
      releaseTimeline.setPhases(phaseTimelines.asJava)
      Option(releaseTimeline)
    case _ => None
  }

  private def toPhaseTimeline(plan: Computed): Option[PhaseTimeline] = plan match {
    case phasePlan: Container =>
      Option(new PhaseTimeline(phasePlan.item.item.asInstanceOf[Phase], phasePlan.start.toDate, phasePlan.end.toDate))
    case _ => None
  }

  private def getReleaseEndDate(plan: Container, phaseTimelines: Seq[PhaseTimeline]): Date = {
    // planner currently doesn't calculate correctly the release end date in case of cyclical dependencies, this helps a bit
    val plannedReleaseEndDate = plan.end.toDate
    val lastPhaseEndDate = phaseTimelines.lastOption.map(_.getPlannedEndDate)
    if (lastPhaseEndDate.isDefined && lastPhaseEndDate.get.after(plannedReleaseEndDate)) lastPhaseEndDate.get else plannedReleaseEndDate
  }


  implicit def dateOrdering[A <: DateTime]: Ordering[A] = Ordering.by(_.getMillis)

}
