package com.xebialabs.xlrelease.reports.api.internal;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.StreamingOutput;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.api.utils.ResponseHelper;
import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters;
import com.xebialabs.xlrelease.api.v1.forms.TimeFrame;
import com.xebialabs.xlrelease.param.DateParam;
import com.xebialabs.xlrelease.plugins.dashboard.domain.Tile;
import com.xebialabs.xlrelease.plugins.dashboard.service.TileService;
import com.xebialabs.xlrelease.reports.audit.MultiReleaseAuditReport;
import com.xebialabs.xlrelease.reports.audit.MultiReleaseAuditReportComponent;
import com.xebialabs.xlrelease.reports.audit.UserPermissionsReport;
import com.xebialabs.xlrelease.reports.excel.ReportExcel;
import com.xebialabs.xlrelease.reports.filters.ReportFilter;
import com.xebialabs.xlrelease.reports.service.ReportParams;
import com.xebialabs.xlrelease.reports.timeline.TimelineExport;
import com.xebialabs.xlrelease.reports.timeline.TimelineExportComponent;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.udm.reporting.filters.impl.DateFilter;

import static com.xebialabs.xlrelease.api.v1.forms.TimeFrame.DEFAULT_TIME_FRAME;
import static com.xebialabs.xlrelease.reports.utils.ReportUtils.splitTags;

/**
 * Provides release exports in various formats.
 */
@Path("/export")
@Controller
public class ExportReportsResource {

    private PermissionChecker permissions;
    private ReportExcel reportExcel;
    private TileService tileService;
    private final ReportResource reportResource;
    private TimelineExportComponent timelineExportComponent;

    @Autowired
    public ExportReportsResource(PermissionChecker permissions,
                                 ReportExcel reportExcel,
                                 TileService tileService,
                                 ReportResource reportResource,
                                 TimelineExportComponent timelineExportComponent
    ) {
        this.permissions = permissions;
        this.reportExcel = reportExcel;
        this.tileService = tileService;
        this.reportResource = reportResource;
        this.timelineExportComponent = timelineExportComponent;
    }

    @GET
    @Path("report")
    @Produces(MediaType.APPLICATION_OCTET_STREAM)
    public Response exportReportToExcel(@DefaultValue(DEFAULT_TIME_FRAME) @QueryParam("since") TimeFrame timeFrame,
                                        @QueryParam("report") ReportResource.ReportType report,
                                        @QueryParam("tags") String tags,
                                        @QueryParam("from") @DateParam Date from,
                                        @QueryParam("to") @DateParam Date to) {
        ReportParams reportParams = ReportParams.apply(timeFrame, from, to, splitTags(tags), new ArrayList<>(), true, null, true);
        Workbook workbook = reportExcel.getWorkBook(report, reportParams);

        StreamingOutput output = workbook::write;

        return ResponseHelper.streamFile(getReportFileName(report), output, MultiReleaseAuditReportComponent.EXCEL_CONTENT_TYPE());
    }

    @GET
    @Path("tile-report")
    @Produces(MediaType.APPLICATION_OCTET_STREAM)
    public Response exportTileReportToExcel(@QueryParam("report") ReportResource.ReportType report,
                                            @QueryParam("tileId") String tileId) {
        Tile tile = tileService.findTileById(tileId);
        List<ReportFilter> filters = tile.getProperty("filters");
        if (filters != null) {
            DateFilter dateFilter = (DateFilter) filters.stream()
                    .filter(filter -> filter.getType().instanceOf(Type.valueOf(DateFilter.class)))
                    .findFirst()
                    .orElse(new DateFilter(TimeFrame.LAST_SIX_MONTHS));

            ReportParams reportParams = ReportParams.apply(dateFilter.getTimeFrame(), dateFilter.getFrom(), dateFilter.getTo(),
                    null, filters, false, null, true);
            Workbook workbook = reportExcel.getWorkBook(report, reportParams);

            StreamingOutput output = workbook::write;

            return ResponseHelper.streamFile(getReportFileName(report), output, MultiReleaseAuditReportComponent.EXCEL_CONTENT_TYPE());
        }

        return Response.noContent().build();
    }

    @POST
    @Path("audit-report/multirelease")
    @Produces(MediaType.APPLICATION_JSON)
    public Response exportMultiReleaseAuditReportToExcel(MultiReleaseReportForm multiReleaseReportForm) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        MultiReleaseAuditReport reportDefinition = new MultiReleaseAuditReport();
        reportDefinition.setFilters(multiReleaseReportForm.getReportFilters());
        return reportResource.submit(reportDefinition);
    }

    @POST
    @Path("audit-report/multirelease-preview")
    @Produces(MediaType.APPLICATION_JSON)
    public Response previewMultiReleaseAuditReportToExcel(MultiReleaseReportForm multiReleaseReportForm) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        MultiReleaseAuditReport reportDefinition = new MultiReleaseAuditReport();
        reportDefinition.setFilters(multiReleaseReportForm.getReportFilters());
        return reportResource.preview(reportDefinition);
    }

    @POST
    @Path("audit-report/user-permissions")
    @Produces(MediaType.APPLICATION_JSON)
    public Response exportUserPermissionsAuditReportToExcel(UserPermissionsReportForm userPermissionsReportForm) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        UserPermissionsReport reportDefinition = new UserPermissionsReport();
        reportDefinition.setFilters(userPermissionsReportForm.getReportFilters());
        return reportResource.submit(reportDefinition);
    }

    @POST
    @Path("audit-report/user-permissions-preview")
    @Produces(MediaType.APPLICATION_JSON)
    public Response previewUserPermissionsAuditReportToExcel(UserPermissionsReportForm userPermissionsReportForm) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        UserPermissionsReport reportDefinition = new UserPermissionsReport();
        reportDefinition.setFilters(userPermissionsReportForm.getReportFilters());
        return reportResource.preview(reportDefinition);
    }

    private String getReportFileName(ReportResource.ReportType report) {
        return report.name().toLowerCase() + MultiReleaseAuditReportComponent.EXCEL_EXTENSION();
    }

    @POST
    @Path("timeline")
    @Produces(MediaType.APPLICATION_JSON)
    public Response exportTimeline(ReleasesFilters releasesFilters) {
        Workbook workbook = timelineExportComponent.exportTimeline(releasesFilters);
        StreamingOutput output = workbook::write;
        return ResponseHelper.streamFile(TimelineExport.EXPORT_FILE_NAME(), output, TimelineExport.EXPORT_CONTENT_TYPE());
    }

}
