package com.xebialabs.xlrelease.reports.service

import com.google.common.cache.{CacheBuilder, CacheLoader, LoadingCache}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.reports.dto._
import com.xebialabs.xlrelease.reports.service.ReportServiceCache._
import com.xebialabs.xlrelease.views.Point
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.concurrent.TimeUnit.MINUTES
import java.util.{List => JList}

object ReportServiceCache {
  val CACHE_NAME: String = "reporting"

  implicit class GetDataWithRefresh[A](val loadingCache: LoadingCache[ReportParams, A]) {
    def getData(params: ReportParams): A = {
      if (params.refresh) {
        loadingCache.invalidate(params)
      }
      loadingCache.get(params)
    }
  }

}

@Service
class ReportServiceCache @Autowired()(reportService: ReportService, xlrConfig: XlrConfig) {

  def builder[A <: Object](f: ReportParams => A): LoadingCache[ReportParams, A] =
    CacheBuilder.newBuilder.maximumSize(xlrConfig.cache.maxSize(CACHE_NAME)).expireAfterWrite(xlrConfig.cache.ttl(CACHE_NAME).toMinutes, MINUTES)
      .build[ReportParams, A](new CacheLoader[ReportParams, A]() {
      override def load(key: ReportParams): A = f(key)
    })

  private val releaseDurationCache =
    builder(p => reportService.getReleaseDuration(p))

  private val completedReleasesCache =
    builder(p => reportService.getCompletedReleases(p))

  private val averageAndLongestReleaseDurationCache =
    builder(p => reportService.getAverageAndLongestReleaseDuration(p))

  private val peopleMostInvolvedCache =
    builder(p => reportService.getTopPeopleMostInvolved(p))

  private val longestTasksCache =
    builder(p => reportService.getTopLongestTasks(p))

  private val longestTaskTypesCache =
    builder(p => reportService.getTopLongestTaskTypes(p))

  private val averageAndLongestTaskDurationCache =
    builder(p => reportService.getAverageAndLongestTaskDuration(p))

  private val releaseAutomationCache =
    builder(p => reportService.getReleaseAutomationSeries(p))

  private val mostRecentReleasesCache =
    builder(p => reportService.getReleaseAutomation(p))

  private val longestPhasesCache =
    builder(p => reportService.getTopLongestPhases(p))

  private val numberOfReleaseCache =
    builder(p => reportService.getNumberOfReleaseByMonth(p))

  //

  def getReleaseDuration(reportParams: ReportParams): ReleasesDuration = releaseDurationCache.getData(reportParams)

  def getCompletedReleases(reportParams: ReportParams): CompletedReleases = completedReleasesCache.getData(reportParams)

  def getAverageAndLongestReleaseDuration(reportParams: ReportParams): AverageAndLongestReleaseDuration =
    averageAndLongestReleaseDurationCache.getData(reportParams)

  def getTopPeopleMostInvolved(reportParams: ReportParams): JList[UserParticipation] = peopleMostInvolvedCache.getData(reportParams)

  def getTopLongestTasks(reportParams: ReportParams): JList[TaskDuration] = longestTasksCache.getData(reportParams)

  def getTopLongestTaskTypes(reportParams: ReportParams): JList[LongestTaskType] = longestTaskTypesCache.getData(reportParams)

  def getAverageAndLongestTaskDuration(reportParams: ReportParams): AverageAndLongestTaskDuration =
    averageAndLongestTaskDurationCache.getData(reportParams)

  def getReleaseAutomationSeries(reportParams: ReportParams): ReleasesAutomation = releaseAutomationCache.getData(reportParams)

  def getTopLongestPhases(reportParams: ReportParams): JList[PhaseDuration] = longestPhasesCache.getData(reportParams)

  def getNumberOfReleaseByMonth(reportParams: ReportParams): JList[Point] = numberOfReleaseCache.getData(reportParams)

  def getMostRecentReleasesData(reportParams: ReportParams): JList[ReleaseAutomationData] = mostRecentReleasesCache.getData(reportParams)

  private val cache = List[LoadingCache[_, _]](
    releaseDurationCache, peopleMostInvolvedCache, longestTasksCache,
    releaseAutomationCache, longestPhasesCache, numberOfReleaseCache,
    mostRecentReleasesCache
  )

  def clearCache(): Unit = {
    cache.foreach(_.invalidateAll())
  }
}
