package com.xebialabs.xlrelease.reports.service

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, Type}
import com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY
import com.xebialabs.deployit.{ReleaseInfo, ServerConfiguration}
import com.xebialabs.xlrelease.api.v1.filter.{ApplicationFilters, EnvironmentFilters}
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.facet.{Facet, TaskReportingRecord}
import com.xebialabs.xlrelease.domain.status.ReleaseStatus.PLANNED
import com.xebialabs.xlrelease.domain.udm.reporting._
import com.xebialabs.xlrelease.domain.utils.{AdaptiveReleaseId, ReleaseIdInDatabase}
import com.xebialabs.xlrelease.environments.repository.{ApplicationRepository, EnvironmentRepository}
import com.xebialabs.xlrelease.planner.{PlannerReleaseItem, PlannerReleaseTree}
import com.xebialabs.xlrelease.reports.audit.CommonFormat.{CompletedBy, PreResolvedProperties, TaskAgent}
import com.xebialabs.xlrelease.reports.audit.TaskDetailsReport.getPropertyValue
import com.xebialabs.xlrelease.reports.audit.TaskRecordReport.TaskRows
import com.xebialabs.xlrelease.reports.audit._
import com.xebialabs.xlrelease.reports.domain.MaybeData._
import com.xebialabs.xlrelease.reports.domain._
import com.xebialabs.xlrelease.reports.domain.exceptions.ItemNotFoundException
import com.xebialabs.xlrelease.reports.excel.{AuditReport, ReleaseReport}
import com.xebialabs.xlrelease.reports.repository.deployment.DeploymentRecordRepository
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.search.ReleaseCountResults
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.{AUDIT_ALL, EDIT_FOLDER_TEAMS}
import com.xebialabs.xlrelease.security.sql.snapshots.service.PermissionsSnapshotService
import com.xebialabs.xlrelease.service._
import com.xebialabs.xlrelease.udm.reporting._
import com.xebialabs.xlrelease.user.User
import com.xebialabs.xlrelease.views.LogsFilters
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.data.domain.Pageable
import org.springframework.stereotype.Service

import java.util.Date
import scala.jdk.CollectionConverters._
import scala.util.Try

object ReportsService {
  val SYSTEM_USER_NAME = "SYSTEM"
  lazy val releaseInfo = new ReleaseInfo("com/xebialabs/deployit/release.properties")
}

@Service
class ReportsService @Autowired()(userInfoResolver: UserInfoResolver,
                                  configurationRepository: ConfigurationRepository,
                                  releaseRepository: ReleaseRepository,
                                  folderRepository: FolderRepository,
                                  triggerRepository: TriggerRepository,
                                  teamRepository: TeamRepository,
                                  activityLogsService: ReleaseActivityLogsService,
                                  releaseSearchService: ReleaseSearchService,
                                  releaseExtensionsRepository: ReleaseExtensionsRepository,
                                  archivingService: ArchivingService,
                                  releaseService: ReleaseService,
                                  serverConfiguration: ServerConfiguration,
                                  facetRepositoryDispatcher: FacetRepositoryDispatcher,
                                  deploymentRecordRepository: DeploymentRecordRepository,
                                  titleService: TitleService,
                                  applicationRepository: ApplicationRepository,
                                  environmentRepository: EnvironmentRepository,
                                  rolesSnapshotService: PermissionsSnapshotService,
                                  permissionChecker: PermissionChecker
                                 ) {
  private lazy val planRecordType: Type = Type.valueOf(classOf[PlanRecord])
  private lazy val buildRecordType: Type = Type.valueOf(classOf[BuildRecord])
  private lazy val codeComplianceRecordType: Type = Type.valueOf(classOf[CodeComplianceRecord])
  private lazy val itsmRecordType: Type = Type.valueOf(classOf[ItsmRecord])
  private lazy val deploymentRecordType: Type = Type.valueOf(classOf[DeploymentRecord])

  def fetchReleaseReportData(release: Release): ReleaseReport.Data = {
    ReleaseReport.Data(release, getReleaseTasksAgents(getReleaseData(release, includeAllLogEntries = false)))
  }

  private def getReleaseData(release: Release, includeAllLogEntries: Boolean): ReleaseData = {
    val lifecycleFilter = if (includeAllLogEntries) {
      LogsFilters.ALL
    } else {
      val f = new LogsFilters()
      f.setLifecycle(true)
      f
    }
    ReleaseData(
      release,
      activityLogsService.getFilteredLogs(release.getId, lifecycleFilter, Pageable.unpaged()).asScala.toSeq
    )
  }

  def fetchAuditReportData(release: Release): AuditReport.Data = {
    val releaseData = getReleaseData(release, includeAllLogEntries = true)
    val releaseTree = PlannerReleaseTree(release)
    val taskRowsByRecordType: Map[Type, Seq[TaskRows]] = fetchRecordReportData(releaseTree, releaseData)
    val taskAgents = getReleaseTasksAgents(releaseData)
    AuditReport.Data(
      releaseTree = releaseTree,
      template = fetchReleaseTemplateTitleAndPath(Option(release.getOriginTemplateId)),
      releaseOverviewData = fetchReleaseOverviewData(releaseTree, taskAgents, releaseData),
      planRecordsData = TaskRecordReport.Data(planRecordType, release, taskRowsByRecordType.getOrElse(planRecordType, Seq.empty)),
      buildRecordsData = TaskRecordReport.Data(buildRecordType, release, taskRowsByRecordType.getOrElse(buildRecordType, Seq.empty)),
      codeComplianceRecordsData = TaskRecordReport.Data(codeComplianceRecordType, release, taskRowsByRecordType.getOrElse(codeComplianceRecordType, Seq.empty)),
      itsmRecordsData = TaskRecordReport.Data(itsmRecordType, release, taskRowsByRecordType.getOrElse(itsmRecordType, Seq.empty)),
      deploymentsData = TaskRecordReport.Data(deploymentRecordType, release, taskRowsByRecordType.getOrElse(deploymentRecordType, Seq.empty)),
      activityLogsData = fetchActivityLogData(releaseData),
      taskInputOutputData = fetchTaskInputOutputData(releaseTree, taskAgents),
      permissionData = fetchPermissionReportData(release)
    )
  }

  def getReleaseIdsFromFilters(request: AuditReportRequest): Seq[ReleaseIdInDatabase] = {
    searchAllIds(request)
  }

  def searchAllIds(request: AuditReportRequest,
                   page: Option[Long] = None,
                   numberByPage: Option[Long] = None
                  ): Seq[ReleaseIdInDatabase] = {
    releaseSearchService.searchAllIdsWithoutPermissionsCheck(request.filters, request.order, page, numberByPage)
  }

  def getReleaseTasksAgents(releaseData: ReleaseData): Map[String, TaskAgent] = {
    val release = releaseData.release
    for {
      task <- release.getAllTasks.asScala
      assigneeAndTeam = {
        val owner = CommonFormat.getTaskOwner(release, task).map(userInfoResolver.getFullNameOrUsernameOf)
        val team = CommonFormat.getTaskTeam(task)
        (owner, team)
      }
      completedBy = getCompletedByFromActivityLogs(releaseData, task)
      resolvedProperties = resolveTaskProperties(task)
    } yield {
      task.getId -> TaskAgent(assigneeAndTeam, completedBy, resolvedProperties)
    }
  }.toMap

  private def getCompletedByFromActivityLogs(releaseData: ReleaseData, task: Task): CompletedBy = {
    for {
      completedByLog <- releaseData.completedByActivityLogEntriesByTask.get(task.getId)
      name <- resolveUser(completedByLog.getUsername)
    } yield name
  }

  private def resolveUser(username: String): Option[String] = Option(username).map(userInfoResolver.getFullNameOrUsernameOf)

  private def resolveTaskProperties(task: Task): PreResolvedProperties = {
    def isInputHintReferenceTypeDelivery(pd: PropertyDescriptor): Boolean = {
      for {
        inputHint <- Option(pd.getInputHint)
        referencedType <- Option(inputHint.getReferencedType)
      } yield referencedType.instanceOf(Type.valueOf("delivery.Delivery"))
    }.getOrElse(false)

    def getPropertyValueAsString(pythonScript: PythonScript, pd: PropertyDescriptor) = {
      getPropertyValue(task.getRelease, pythonScript, pd, task.getVariableMapping.asScala, replaceVariables = true)
        .getOrElse(pythonScript.getProperty(pd.getName))
        .asInstanceOf[String]
    }

    task match {
      case crt: CreateReleaseTask =>
        val maybeOptionalTemplateTitleAndPath = fetchReleaseTemplateTitleAndPath(Option(crt.getTemplateId))
        val maybePath: Maybe[String] = CommonFormat.getMaybeReleasePath(maybeOptionalTemplateTitleAndPath)
        val maybeTitle = CommonFormat.getMaybeReleaseTitle(maybeOptionalTemplateTitleAndPath)
        val templateName = MaybeData.combine(maybePath, maybeTitle)(_ + s" $SEPARATOR " + _)
        val maybeFolderPath = formattedFolderPath(crt.getFolderId)
        Map(
          "templateId" -> templateName,
          "folderId" -> maybeFolderPath
        )
      case t: Task if t.getTaskType.instanceOf(Type.valueOf("delivery.Task")) =>
        val pythonScript = t.asInstanceOf[CustomScriptTask].getPythonScript
        val resolvedProperties = for {
          pd <- pythonScript.getInputProperties.asScala ++ pythonScript.getOutputProperties.asScala
          if isInputHintReferenceTypeDelivery(pd)
          value <- Option(getPropertyValueAsString(pythonScript, pd))
        } yield {
          val title = titleService.getTitleById(value).getOrElse(value)
          pd.getName -> MaybeData(title)
        }
        resolvedProperties.toMap
      case _ => Map.empty
    }
  }

  def getInstanceName(): Option[String] = Option(configurationRepository.getThemeSettings().getHeaderName)
    .filter(_.nonEmpty)

  def getServerUrl(): Option[String] = {
    Option(serverConfiguration.getServerUrl())
  }

  def getInstanceVersion(): Option[String] = {
    Option(ReportsService.releaseInfo.getVersion)
  }

  private def getAuthenticatedUserFullName(): String = userInfoResolver.getFullNameOrUsernameOf(User.AUTHENTICATED_USER.getName)

  def formattedReleaseFolderPath(releaseId: String): Maybe[String] = {
    MaybeData(findFolderId(releaseId)).flatMapMaybe(formattedFolderPath)
  }

  def formattedFolderPath(folderId: String): Maybe[String] = {
    Option(folderId).collect {
      case id if id != "" && id != ROOT_FOLDER_ID =>
        getFolderPath(id).map {
          case Right(parts) => Right(CommonFormat.formatFolderPath(parts))
          case Left((t, parts)) => Left(t -> parts.mkString(", "))
        }
    }.getOrElse(success(""))
  }

  protected def fetchReleaseTemplateTitleAndPath(releaseId: Option[String]): Maybe[Option[ReleaseTitleAndPath]] = {
    releaseId match {
      case None => MaybeData.success(Option.empty[ReleaseTitleAndPath])
      case Some(id) =>
        MaybeData.recoverWith(
          Try(releaseRepository.getTitle(id))
            .map(releaseTitle => Some(ReleaseTitleAndPath(releaseTitle, formattedReleaseFolderPath(id)))),
          t => ItemNotFoundException(s"Template [$id] does not exist", t) ->
            Some(ReleaseTitleAndPath(id, formattedReleaseFolderPath(id)))
        )
    }
  }

  protected def fetchReleaseOverviewData(releaseTree: PlannerReleaseTree,
                                         taskAgents: Map[String, TaskAgent],
                                         releaseData: ReleaseData): ReleaseOverviewReport.Data = {
    val templateInfo = fetchReleaseTemplateTitleAndPath(
      Option(releaseTree.release.getOriginTemplateId).filter(_.nonEmpty))
    ReleaseOverviewReport.Data(
      generatedBy = getAuthenticatedUserFullName(),
      generatedOn = new Date(),
      masterReportName = "",
      instanceData = InstanceData(getInstanceName(), getServerUrl(), getInstanceVersion()),
      releaseTree = releaseTree,
      releaseFolder = formattedReleaseFolderPath(releaseTree.release.getId),
      releaseOwner = Option(releaseTree.release.getOwner).map(userInfoResolver.getFullNameOrUsernameOf),
      startedBy = getStartedBy(releaseData),
      templateInfo = templateInfo,
      tasksAgents = taskAgents
    )
  }

  protected def fetchRecordReportData(releaseTree: PlannerReleaseTree, releaseData: ReleaseData): Map[Type, Seq[TaskRows]] = {
    val records = facetRepositoryDispatcher.findAllFacetsByRelease(releaseTree.release)
    releaseTree.root.getAllTasks().asScala.toList.foldLeft(Map.empty[Type, Seq[TaskRows]]) {
      case (byRecordType, plannedTask) =>
        getTaskRowsByRecordsType(releaseData, plannedTask, records).foldLeft(byRecordType) {
          case (byRecordType1, (recordType, taskRows)) =>
            byRecordType1 + (recordType -> (byRecordType1.getOrElse(recordType, Seq.empty) :+ taskRows))
        }
    }
  }

  protected def getTaskRowsByRecordsType(releaseData: ReleaseData, plannedTask: PlannerReleaseItem, records: Seq[Facet]): Map[Type, TaskRows] = {
    plannedTask.getTask.map { task =>
      val assigneeAndTeam = (CommonFormat.getTaskOwner(releaseData.release, task).map(userInfoResolver.getFullNameOrUsernameOf),
        CommonFormat.getTaskTeam(task))
      val completedBy = getCompletedByFromActivityLogs(releaseData, task)
      records.filter(f => getFolderlessId(f.getTargetId) == getFolderlessId(task.getId) && f.isInstanceOf[TaskReportingRecord]).toList
        .map(_.asInstanceOf[TaskReportingRecord])
        .groupBy(_.getType).map {
        case (recordType, taskRecords) =>
          recordType -> TaskRows(plannedTask, TaskAgent(assigneeAndTeam, completedBy), taskRecords)
      }
    }.getOrElse(Map.empty)
  }

  protected def fetchActivityLogData(releaseData: ReleaseData): ActivityLogReport.Data = {
    val folderId = releaseData.release.findFolderId()
    val teamData = if (isFolderId(folderId)) {
      // if the folder is deleted, return empty map
      Try(teamRepository.getTeams(folderId).asScala.map(t => (t.getId, t.getTeamName)).toMap).getOrElse(Map.empty[String, String])
    } else {
      Map.empty[String, String]
    }
    ActivityLogReport.Data(
      logs = releaseData.allLogEntries.map { logEntry =>
        logEntry -> resolveUser(logEntry.getUsername).getOrElse(ReportsService.SYSTEM_USER_NAME)
      },
      releaseTitle = Option(releaseData.release.getTitle),
      teams = teamData,
      phases = releaseData.release.getPhases.asScala.map(p => (p.getId, p.getTitle)).toMap,
      tasks = releaseData.release.getAllTasks.asScala.map(t => (t.getId, t.getTitle)).toMap
    )
  }

  def fetchTaskInputOutputData(releaseTree: PlannerReleaseTree, taskAgents: Map[String, TaskAgent]): TaskDetailsReport.Data = {
    TaskDetailsReport.Data(releaseTree, taskAgents)
  }

  def getFolderPath(folderId: String): Maybe[Seq[String]] = {
    MaybeData.recoverWith(
      Try(folderRepository.getPath(folderId)).filter(_.nonEmpty),
      t => ItemNotFoundException(s"Folder Path: [$folderId]", t) -> Seq(folderId)
    )
  }

  protected def getStartedBy(releaseData: ReleaseData): Option[ReleaseStartedBy] = CreatedByDetailsHelper.getCreationDetails(
    releaseRepository, triggerRepository, activityLogsService, resolveUser
  )(releaseData)

  def findReleaseByReleaseIdInDatabase(releaseIdInDatabase: ReleaseIdInDatabase): Release = {
    releaseIdInDatabase match {
      case ReleaseIdInDatabase(id, true, false) =>
        archivingService.getRelease(id.withOnlyOneParentOrApplicationsForArchiveDb(), includePreArchived = true)
      // for the case when release was seen in live db, we make a try to fetch it from live db
      // and fall back to archive db if it is not in live any more
      case ReleaseIdInDatabase(id, true, true) => findReleaseInNonArchiveThenInArchive(id)
      case ReleaseIdInDatabase(id, false, _) => findReleaseInNonArchiveThenInArchive(id)
    }
  }

  private def findReleaseInNonArchiveThenInArchive(releaseIdWithFolderName: AdaptiveReleaseId) = {
    try {
      releaseService.findById(releaseIdWithFolderName.folderlessReleaseId())
    } catch {
      case _: NotFoundException => releaseService.findByIdInArchive(releaseIdWithFolderName.withOnlyOneParentOrApplicationsForArchiveDb())
    }
  }

  def getReleasesForPreview(request: AuditReportRequest, page: Long, numberByPage: Long): java.util.List[Release] = {
    val releaseIds = searchAllIds(request, Some(page), Some(numberByPage))
    getReleasesForPreviewByReleaseIds(releaseIds)
  }

  def getReleasesForPreviewByReleaseIds(releaseIds: Seq[ReleaseIdInDatabase]): java.util.List[Release] = {
    val archiveReleaseIds: Seq[AdaptiveReleaseId] = releaseIds
      .collect { case id if !id.inNonArchive => id.adaptiveReleaseId }
    val nonArchiveReleaseIds: Seq[AdaptiveReleaseId] = releaseIds
      .collect { case id if id.inNonArchive => id.adaptiveReleaseId }
    val archiveReleases: Map[String, Release] = archivingService
      .searchReleasesByReleaseIds(archiveReleaseIds.map(_.withOnlyOneParentOrApplicationsForArchiveDb()))
      .map(r => getFolderlessId(r.getId) -> r)
      .toMap
    val nonArchiveReleases: Map[String, Release] = releaseSearchService
      .searchReleasesByReleaseIds(nonArchiveReleaseIds.map(_.folderlessReleaseId()))
      .map(releaseExtensionsRepository.decorate)
      .map(r => getFolderlessId(r.getId) -> r)
      .toMap
    val missingNonArchiveReleases: Map[String, Release] = nonArchiveReleaseIds
      .collect { case id if !(nonArchiveReleases contains id.folderlessReleaseId()) =>
        id.folderlessReleaseId() -> findReleaseInNonArchiveThenInArchive(id)
      }.toMap
    releaseIds
      .flatMap {
        case id if id.inNonArchive =>
          nonArchiveReleases
            .get(id.adaptiveReleaseId.folderlessReleaseId())
            .orElse(missingNonArchiveReleases.get(id.adaptiveReleaseId.folderlessReleaseId()))
        case id =>
          archiveReleases.get(id.adaptiveReleaseId.folderlessReleaseId())
      }.asJava
  }

  def getReleasesCountByStatus(reportRequest: AuditReportRequest): ReleaseCountResults =
    releaseSearchService.countReleases(reportRequest.filters)

  def getAllApplicationNames: Set[String] = deploymentRecordRepository.findAllApplicationNames ++
    applicationRepository.searchApplications(new ApplicationFilters(), Page.default).map(_.getTitle).toSet

  def getAllEnvironmentNames: Set[String] = deploymentRecordRepository.findAllEnvironmentNames ++
    environmentRepository.searchEnvironments(new EnvironmentFilters(), Page.default).map(_.getTitle).toSet

  def fetchPermissionReportData(release: Release): ReleasePermissionReport.Data = {
    val containerId = getContainerId(release.getId)
    val now = new Date()
    val startDate: Date = release.getStatus match {
      case PLANNED => now
      case _ => release.getStartOrScheduledDate
    }
    val endDate = Option(release.getEndDate).getOrElse(now)

    // No Authentication context during report generation
    if (permissionChecker.isNotAuthenticated ||
      permissionChecker.hasPermission(AUDIT_ALL, release) || permissionChecker.hasPermission(EDIT_SECURITY, release) || permissionChecker.hasPermission(EDIT_FOLDER_TEAMS, release)) {
      val (permissionDataFromSnapshots, snapshotErrors) = rolesSnapshotService.getMergedContainerSnapshotDataForReport(containerId, startDate, endDate)
      val sortedSnapshots = permissionDataFromSnapshots
        .map(container => container.copy(permission = PermissionLabels.getLabel(container.permission)))
        .sortBy(containerPermissionAndPrincipal => (containerPermissionAndPrincipal.principal.toLowerCase, containerPermissionAndPrincipal.permission.toLowerCase))
      ReleasePermissionReport.Data(Some(sortedSnapshots), Some(snapshotErrors))
    } else {
      ReleasePermissionReport.Data(None, None)
    }
  }

  private def getContainerId(id: String): String = if (Ids.isInFolder(id)) {
    Ids.findFolderId(id)
  } else {
    id
  }
}

