package com.xebialabs.xlrelease.reports.job.impl

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.reports.job.api.ReportingEngineService.ReportJobId
import com.xebialabs.xlrelease.reports.job.api.{ReportResult, ReportStorage, StreamingReportResult}
import com.xebialabs.xlrelease.reports.job.domain.ReportJob
import com.xebialabs.xlrelease.reports.job.impl.GlobalReportStorage.{FileReportResult, FileStreamingReportResult, FileSystemReportStorage}
import com.xebialabs.xlrelease.utils.CloseableUtils.using
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils
import org.springframework.util.StreamUtils.copy

import java.io.{File, FileNotFoundException, OutputStream}
import java.net.URI
import java.text.SimpleDateFormat
import java.util.Date
import scala.annotation.tailrec

class GlobalReportStorage(reportStorageLocation: String) extends Logging {


  val rootDir: File = new File(reportStorageLocation)

  if (!rootDir.exists) {
    try {
      rootDir.mkdirs()
      logger.debug(s"Created root directory for reports at '${rootDir.getAbsolutePath}' based on provided location '$reportStorageLocation'")
    } catch {
      case e: Exception =>
        val msg = s"Unable to create root directory for reports '$reportStorageLocation'."
        throw new IllegalStateException(msg, e)
    }
  }

  if (!rootDir.isDirectory) {
    val msg = s"Report storage root directory location '$reportStorageLocation' does not point to a directory."
    throw new IllegalStateException(msg)
  }

  def reportStorage(jobId: ReportJobId, generatedOn: Date): ReportStorage = {
    val reportWorkdir: File = getReportDirectory(jobId, generatedOn)
    if (!reportWorkdir.exists) {
      val isCreated = try {
        reportWorkdir.mkdirs()
      } catch {
        case e: Exception =>
          val msg = s"Unable to create directory for report job '$jobId' in '$rootDir'."
          throw new IllegalStateException(msg, e)
      }
      if (!isCreated) {
        throw new IllegalStateException(s"Unable to create directory for report job '$jobId' in '$rootDir'.")
      }
    }
    logger.debug(s"Created root directory for reports at '${reportWorkdir.getAbsolutePath}' based on provided location '$rootDir'")
    new FileSystemReportStorage(reportWorkdir)
  }

  def resolve(reportResult: ReportResult): StreamingReportResult = {
    reportResult match {
      case streamingReportResult: StreamingReportResult => streamingReportResult
      case _ =>
        val jobStorage = reportStorage(reportResult.jobId, reportResult.generatedOn)
        val jobDir = jobStorage.workdir
        if (!jobDir.exists()) {
          throw new IllegalStateException(s"Report directory of job '${reportResult.jobId}' does not exist.")
        }
        new FileStreamingReportResult(jobDir.toURI, reportResult)
    }
  }

  def reportResult(reportJob: ReportJob): ReportResult = {
    new FileReportResult(reportJob.getJobId(), reportJob.getReportName, reportJob.getEndTime, reportJob.getResultUri, reportJob.getReportDefinition.contentType)
  }

  def getReportDirectory(jobId: ReportJobId, generatedOn: Date): File = {
    val formatter = new SimpleDateFormat("yyyy/MM/dd")
    val dateFragment = formatter.format(generatedOn)
    val reportWorkdir: File = new File(rootDir, s"$dateFragment/$jobId")
    reportWorkdir
  }

  def cleanup(reportJob: ReportJob): Boolean = {
    try {
      val jobDir = getReportDirectory(reportJob.getJobId(), reportJob.getEndTime)

      if (jobDir.exists()) {
        logger.debug(s"Deleting directory: ${jobDir.getPath}")
        FileUtils.deleteDirectory(jobDir)
      }

      try {
        deleteParentsIfEmpty(jobDir)
      } catch {
        case t: Throwable =>
          logger.warn(s"Exception while deleting report's parent directory: ${t.toString}", t)
      }

      true
    } catch {
      case e: Exception =>
        logger.warn(s"Exception while deleting report's job directory: ${e.getMessage}")
        false
    }
  }

  @tailrec
  private final def deleteParentsIfEmpty(dir: File): Unit = {
    val parent = dir.getParentFile
    if (!parent.equals(rootDir) && parent.isDirectory && parent.list().isEmpty) {
      logger.trace(s"Deleting directory: ${parent.getPath}")
      FileUtils.deleteDirectory(parent)
      deleteParentsIfEmpty(parent)
    }
  }

}

object GlobalReportStorage {

  class FileSystemReportStorage(override val workdir: File) extends ReportStorage

  private class FileStreamingReportResult(contextUri: URI, reportResult: ReportResult) extends StreamingReportResult with Logging {

    override def write(outputStream: OutputStream): Unit = {
      try {
        using(resolvedUri.toURL.openStream()) { inputStream =>
          copy(inputStream, outputStream)
        }
      } catch {
        case _: FileNotFoundException =>
          throw new NotFoundException(s"'%s' was not found", reportResult.resultUri)
        case ex: Exception =>
          logger.error(s"Unable to access '${reportResult.resultUri}'", ex)
      }
    }

    lazy val resolvedUri: URI = {
      val unresolvedUri = new URI(reportResult.resultUri)
      if (!unresolvedUri.isAbsolute) {
        contextUri.resolve(unresolvedUri)
      } else {
        unresolvedUri
      }
    }

    override def resultUri: String = resolvedUri.toString

    override def generatedOn: Date = reportResult.generatedOn

    override def reportName: String = reportResult.reportName

    override def contentType: String = reportResult.contentType

    override def jobId: ReportJobId = reportResult.jobId
  }

  private class FileReportResult(override val jobId: ReportJobId,
                         override val reportName: String,
                         override val generatedOn: Date, // instant when report generation is completed ( = end time)
                         override val resultUri: String,
                         override val contentType: String)
    extends ReportResult


}
