package com.xebialabs.xlrelease.reports.job.impl.cleanup

import com.xebialabs.xlrelease.reports.job.impl.ReportExecutorService
import com.xebialabs.xlrelease.reports.job.impl.cleanup.ReportCleanerActor.{CleanupDone, NodeCleanup}
import grizzled.slf4j.Logging
import org.apache.pekko.actor.{Actor, Props, Stash}
import org.apache.pekko.cluster.Member

import scala.concurrent.{ExecutionContextExecutorService, Future}
import scala.util.{Failure, Success}

object ReportCleanerActor {

  sealed trait ReportCleanerCommand

  case object CleanupDone extends ReportCleanerCommand

  case class NodeCleanup(node: Member) extends ReportCleanerCommand

  val name: String = "reportCleaner"

  def props(reportCleanerOperations: ReportCleanerOperations, reportExecutor: ReportExecutorService): Props = {
    Props(new ReportCleanerActor(reportCleanerOperations, reportExecutor))
  }
}

class ReportCleanerActor(reportCleanerOperations: ReportCleanerOperations, reportExecutor: ReportExecutorService)
  extends Actor with Logging with Stash {

  override def receive: Receive = waiting

  // TODO cleanup should be executed on it's own thread pool
  implicit val ec: ExecutionContextExecutorService = reportExecutor.executionContext

  private def waiting: Receive = {
    case NodeCleanup(node) =>
      context.become(working)
      cleanupNode(node)
    case msg => logger.warn(s"Unexpected message received $msg")
  }

  private def working: Receive = {
    case CleanupDone =>
      unstashAll()
      context.become(waiting, discardOld = false)
    case NodeCleanup(_) =>
      logger.debug(s"Node cleanup received while cleaning storage")
      stash()
  }

  private def cleanupNode(node: Member): Unit = {
    val host = node.address.host.get
    val port = node.address.port.get
    val nodeName = s"$host:$port"
    Future {
      logger.info(s"Cleanup of reports after node '$nodeName' was removed from cluster started")
      reportCleanerOperations.nodeCleanup(nodeName)
      logger.info(s"Cleanup of reports after node '$nodeName' was removed from cluster ended")
    } onComplete {
      case Success(_) => self ! CleanupDone
      case Failure(ex) =>
        logger.error(s"Exception occurred while cleaning reports after node '$nodeName' was removed from cluster", ex)
        self ! CleanupDone
    }
  }

}


